﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Threading.Tasks;
using Microsoft.Azure.ActiveDirectory.GraphClient;
using Microsoft.Azure.ActiveDirectory.GraphClient.Extensions;
using Microsoft.IdentityModel.Clients.ActiveDirectory;

namespace CCSAdmin
{
    public class CcsAdmin
    {
        private CcsAdmin()
        {
        }
        private static readonly Lazy<CcsAdmin> CcsAdminLazy = new Lazy<CcsAdmin>(() => new CcsAdmin(), true);
        public static CcsAdmin Instance
        {
            get { return CcsAdminLazy.Value; }
        }

        private static readonly object MInitLock = new object();

        public void InitAdminCredentials(
            string clientId,
            string appKey,
            string aadAuthRoot,  /*https://login.windows.net*/
            string graphApiRoot, /*"https://graph.windows.net"*/
            string aadTenantName, /*the AAD tenant name from your subscription*/
            string defaultRoleForNewUser /*Consumers*/)
        {
            if (AdminAadClient == null)
            {
                lock (MInitLock)
                {
                    if (AdminAadClient == null)
                    {
                        //sth like: "https://login.windows.net/wangchuang1103hotmail.onmicrosoft.com";
                        _mAuthString = string.Format(AuthStringFormat, aadAuthRoot.Trim().TrimEnd('/'),
                            aadTenantName.Trim());
                        _mAppId = clientId;
                        _mAppKey = appKey;
                        _mAadTenantName = aadTenantName;
                        _mGraphApiRoot = graphApiRoot;
                        _mDefaultRoleForNewUser = defaultRoleForNewUser;
                        _mAadTenantGraphApiRoot =
                            new Uri(string.Format(AadGraphApiFormat, graphApiRoot.Trim().TrimEnd('/'),
                                aadTenantName.Trim()));

                        AdminAadClient = new ActiveDirectoryClient(
                            _mAadTenantGraphApiRoot,
                            async () => await GetAppTokenAsync());

                        DomainName = aadTenantName;
                    }
                }
            }
        }

        private string _mAuthString;  // e.g. "https://login.windows.net/wangchuang1103hotmail.onmicrosoft.com";
        private const string AuthStringFormat = "{0}/{1}/"; //aadAuthRoot/aadTenantName

        // These are the credentials the application will present during authentication
        private string _mAppId;
        private string _mAppKey;
        private string _mAadTenantName;
        private string _mDefaultRoleForNewUser;
        private string _mGraphApiRoot;
        // The Azure AD Graph API is the "resource" we're going to request access to.
        private Uri _mAadTenantGraphApiRoot;// e.g.  "https://graph.windows.net/wangchuang1103hotmail.onmicrosoft.com";
        private const string AadGraphApiFormat = "{0}/{1}/"; //graphAPIRoot/aadTenant


        private const string UserNameFormat = "ccs-{0}";
        private const string PrincipalNameFormat = "ccs-{0}@{1}"; //ccs-mobile@aadtentname
        
        private AppRole _mAppRoleForNewUser;
        private ServicePrincipal _mServicePrincipal;

        private ActiveDirectoryClient AdminAadClient { get; set; }

        private async Task<ServicePrincipal> GetServicePrincipalAsync()
        {
            if (_mServicePrincipal == null)
            {
                _mServicePrincipal = (ServicePrincipal)AdminAadClient.ServicePrincipals.Where(isp => isp.AppId == _mAppId).ExecuteSingleAsync().Result;
            }
            System.Diagnostics.Debug.Assert(_mServicePrincipal != null);
            return _mServicePrincipal;
        }

        private async Task<AppRole> GetRole(string roleName)
        {
            var sp = await GetServicePrincipalAsync();
            var appRole = sp.AppRoles.Where(role => role.DisplayName == roleName).FirstOrDefault();
            if (appRole == null)
            {
                throw new Exception("No such app role!");
            }
            return appRole;
         }
        public string DomainName { get; private set; }

        private async Task<string> GetAppTokenAsync()
        {
            // Instantiate an AuthenticationContext for my directory (see authString above).
            AuthenticationContext authenticationContext = new AuthenticationContext(_mAuthString, false);

            // Create a ClientCredential that will be used for authentication.
            // This is where the Client ID and Key/Secret from the Azure Management Portal is used.
            ClientCredential clientCred = new ClientCredential(_mAppId, _mAppKey);

            // Acquire an access token from Azure AD to access the Azure AD Graph (the resource)
            // using the Client ID and Key/Secret as credentials.
            AuthenticationResult authenticationResult = await authenticationContext.AcquireTokenAsync(_mGraphApiRoot, clientCred);

            // Return the access token.
            return authenticationResult.AccessToken;
        }

        /// <summary>
        /// This method does not use ExecuteSingleAsync, instead we use ExecuteAsync, because 
        /// ExecuteSingleAsync returns: System.NullReferenceException: Object reference not set to an instance of an object
        /// 
        /// </summary>
        /// <param name="mobileNumber"></param>
        /// <returns>Null if a user cannot be found, otherwise return the user that its UPN matches the mobile number</returns>
        public async Task<IUser> FindUserByMobileNumberAsync(string mobileNumber)
        {
            if (AdminAadClient == null)
                throw new ApplicationException("Admin aad client is not initialized");
            if (string.IsNullOrWhiteSpace(mobileNumber))
                throw new ArgumentNullException("mobileNumber");

            string userPrincipalName = GetPrincipalNameFromMobile(mobileNumber);

            IPagedCollection<IUser> pagedUsers = await AdminAadClient.Users.Where(
                user => user.UserPrincipalName.Equals(
                    userPrincipalName, StringComparison.CurrentCultureIgnoreCase)).ExecuteAsync();

            IReadOnlyList<IUser> matchedUsers = pagedUsers.CurrentPage;

            if (matchedUsers == null || matchedUsers.Count == 0)
                return null; //user not found

            if (matchedUsers.Count != 1)
                throw new Exception("Find User by UPN returns more than 1 records");

            return matchedUsers.First();
        }

        public IUser FindUserByMobileNumber(string mobileNumber)
        {
            if (AdminAadClient == null)
                throw new ApplicationException("Admin aad client is not initialized");
            if (string.IsNullOrWhiteSpace(mobileNumber))
                throw new ArgumentNullException("mobileNumber");

            string userPrincipalName = GetPrincipalNameFromMobile(mobileNumber);

            IPagedCollection<IUser> pagedUsers = AdminAadClient.Users.Where(
                user => user.UserPrincipalName.Equals(
                    userPrincipalName, StringComparison.CurrentCultureIgnoreCase)).ExecuteAsync().Result;

            IReadOnlyList<IUser> matchedUsers = pagedUsers.CurrentPage;

            if (matchedUsers == null || matchedUsers.Count == 0)
                return null; //user not found

            if (matchedUsers.Count != 1)
                throw new Exception("Find User by UPN returns more than 1 records");

            return matchedUsers.First();
        }


        /// <summary>
        /// This call is not asychronized call. Because:
        /// When user base is large, it is impossible to hold all IUser object in memory,
        /// intead we need use yield return to return back user one by one.
        /// 
        /// If use await, the return type cannot be IEnumerable
        /// </summary>
        /// <returns></returns>
        public IEnumerable<IUser> FindUsers()
        {
            if (AdminAadClient == null)
                throw new ApplicationException("Admin aad client is not initialized");

            IPagedCollection<IUser> pagedCollection = AdminAadClient.Users.ExecuteAsync().Result;
            if (pagedCollection != null)
            {
                do
                {
                    List<IUser> pageUserList = pagedCollection.CurrentPage.ToList();
                    foreach (IUser user in pageUserList)
                    {
                        yield return user;
                    }
                    pagedCollection = pagedCollection.GetNextPageAsync().Result;
                } while (pagedCollection != null);
            }
        }

        /// <summary>
        /// Required parameters when creating user
        /// https://msdn.microsoft.com/Library/Azure/Ad/Graph/api/users-operations#CreateUsers
        /// accountEnabled	boolean	true if the account is enabled; otherwise, false.
        /// displayName	string	The name to display in the address book for the user.
        /// immutableId	string	Only needs to be specified when creating a new user account if you are using a federated domain for the user's userPrincipalName (UPN) property.
        /// mailNickname	string	The mail alias for the user.
        /// passwordProfile	PasswordProfile	The password profile for the user.
        /// userPrincipalName	string	The user principal name (someuser@contoso.com).
        /// </summary>
        /// <param name="password"></param>
        /// <param name="mobileNumber"></param>
        /// <returns></returns>
        /// 
        public async Task<string> CreateUserAsync(string mobileNumber, string password)
        {
            System.Diagnostics.Debug.Assert(false);
            throw new ApplicationException("this should be depreciated!");
            if (AdminAadClient == null)
                throw new ApplicationException("Admin aad client is not initialized");
            if (string.IsNullOrWhiteSpace(mobileNumber))
                throw new ArgumentNullException("mobileNumber");
            if (string.IsNullOrWhiteSpace(password))
                throw new ArgumentNullException("password");


            IUser userInAdd = await FindUserByMobileNumberAsync(mobileNumber);
            if (userInAdd != null)
                throw new ApplicationException(string.Format("a user with mobile number {0} already exists", mobileNumber));

            var newUser = new User
            {
                // Required settings
                DisplayName = string.Format(UserNameFormat, mobileNumber),
                UserPrincipalName =  GetPrincipalNameFromMobile(mobileNumber), 
                PasswordProfile = new PasswordProfile
                {
                    Password = password,
                    ForceChangePasswordNextLogin = false
                },
                MailNickname = mobileNumber,
                AccountEnabled = true,

                // Some (not all) optional settings
                Mobile = mobileNumber,
            };
            
            
            await AdminAadClient.Users.AddUserAsync(newUser);


            // Update the user's role (once we have user's ObjectId)
            await UpdateRoleAsync(newUser, _mDefaultRoleForNewUser);
       
            return newUser.UserPrincipalName;
        }

        public async Task UpdateRoleAsync(User user, string role)
        {
            var appRole = await GetRole(role);
            var sp = await GetServicePrincipalAsync();
            AppRoleAssignment appRoleAssignment = new AppRoleAssignment
            {
                Id = appRole.Id,
                ResourceId = Guid.Parse(sp.ObjectId),
                PrincipalId = Guid.Parse(user.ObjectId)          
            };

            user.AppRoleAssignments.Add(appRoleAssignment);

            await user.UpdateAsync();
        }

#if false  // deprecated
        public string CreateUser(string mobileNumber, string password)
        {
            if (AdminAadClient == null)
                throw new ApplicationException("Admin aad client is not initialized");
            if (string.IsNullOrWhiteSpace(mobileNumber))
                throw new ArgumentNullException("mobileNumber");
            if (string.IsNullOrWhiteSpace(password))
                throw new ArgumentNullException("password");


            IUser userInAdd = FindUserByMobileNumber(mobileNumber);
            if (userInAdd != null)
                throw new ApplicationException(string.Format("a user with mobile number {0} already exists", mobileNumber));

            var newUser = new User
            {
                // Required settings
                DisplayName = string.Format(UserNameFormat, mobileNumber),
                UserPrincipalName = string.Format(PrincipalNameFormat, mobileNumber, _mAadTenantName),
                PasswordProfile = new PasswordProfile()
                {
                    Password = password,
                    ForceChangePasswordNextLogin = false
                },
                MailNickname = mobileNumber,
                AccountEnabled = true,

                // Some (not all) optional settings
                Mobile = mobileNumber,
            };
            // Add the user to the directory
            AdminAadClient.Users.AddUserAsync(newUser).Wait();
            return newUser.UserPrincipalName;
        }
#endif
        
        public async Task<string> CreateUserAsync(string username, 
            string password, IUser userParameters, string role, string domainName = "")
        {
            if (string.IsNullOrWhiteSpace(domainName))
            {
                domainName = _mAadTenantName;
            }
            if (AdminAadClient == null)
            {
                throw new ApplicationException("Admin AAD client is not initialized");
            }
            if (string.IsNullOrWhiteSpace(username))
            {
                throw new ArgumentNullException("username");
            }
            if (string.IsNullOrWhiteSpace(password))
            {
                throw new ArgumentNullException("password");
            }
            
            IUser userInAdd = FindUser(username);
            if (userInAdd != null)
            {
                throw new ApplicationException(string.Format("user name {0} already exists", username));
            }
            var newUser = (userParameters != null) ? userParameters : new User();
            newUser.DisplayName = username;
            newUser.UserPrincipalName = string.Format("{0}@{1}", username, domainName);
            newUser.PasswordProfile = new PasswordProfile
            {
                Password = password,
                ForceChangePasswordNextLogin = false
            };
            newUser.MailNickname = username;
            newUser.AccountEnabled = true;
            // Add the user to the directory
            await AdminAadClient.Users.AddUserAsync(newUser);

            if (!string.IsNullOrWhiteSpace(role))
            {
                await UpdateRoleAsync(newUser as User, role);
            }
            
            return newUser.UserPrincipalName;
        }

        private IUser FindUser(string username)
        {
            if (AdminAadClient == null)
            {
                throw new ApplicationException("Admin AAD client is not initialized");
            }
            if (string.IsNullOrWhiteSpace(username))
            {
                throw new ArgumentNullException("username");
            }

            string userPrincipalName = string.Format("{0}@{1}", username, _mAadTenantName);

            IPagedCollection<IUser> pagedUsers = AdminAadClient.Users.Where(
                user => user.UserPrincipalName.Equals(
                    userPrincipalName, StringComparison.CurrentCultureIgnoreCase)).ExecuteAsync().Result;

            IReadOnlyList<IUser> matchedUsers = pagedUsers.CurrentPage;

            if (matchedUsers == null || matchedUsers.Count == 0)
            {
                return null; //user not found
            }
            if (matchedUsers.Count != 1)
            {
                throw new Exception("Find User by UPN returns more than 1 records");
            }
            return matchedUsers.First();
        }

        public async Task<string> UpdateUserAsync(string mobileNumber, string password)
        {
            if (AdminAadClient == null)
                throw new ApplicationException("Admin aad client is not initialized");

            if (string.IsNullOrWhiteSpace(mobileNumber))
                throw new ArgumentNullException("mobileNumber");
            if (string.IsNullOrWhiteSpace(password))
                throw new ArgumentNullException("password");

            IUser existUser = await FindUserByMobileNumberAsync(mobileNumber);
            if (existUser == null)
                throw new GraphException(HttpStatusCode.NotFound,
                    string.Format("The user with mobile: {0} has not existed yet", mobileNumber));

            existUser.AccountEnabled = true;
            existUser.PasswordProfile = new PasswordProfile
            {
                Password = password,
                ForceChangePasswordNextLogin = false
            };

            // Add the user to the directory
            await existUser.UpdateAsync();

            return existUser.UserPrincipalName;
        }

        public async Task DisableUserAsync(string username)
        {
            var user = FindUser(username);
            if (user == null)
            {
                throw new GraphException(HttpStatusCode.NotFound, string.Format("[{0}] does not exist", username));
            }
            user.AccountEnabled = false;
            await user.UpdateAsync();
        }

#if false // deprecated
        public string UpdateUser(string mobileNumber, string password)
        {
            if (AdminAadClient == null)
                throw new ApplicationException("Admin aad client is not initialized");

            if (string.IsNullOrWhiteSpace(mobileNumber))
                throw new ArgumentNullException("mobileNumber");
            if (string.IsNullOrWhiteSpace(password))
                throw new ArgumentNullException("password");

            IUser existUser = FindUserByMobileNumber(mobileNumber);
            if (existUser == null)
                throw new GraphException(HttpStatusCode.NotFound, string.Format("The user with mobile: {0} has not existed yet", mobileNumber));

            existUser.AccountEnabled = true;
            existUser.PasswordProfile = new PasswordProfile
            {
                Password = password,
                ForceChangePasswordNextLogin = false
            };
            
            // Add the user to the directory
            existUser.UpdateAsync().Wait();
            return existUser.UserPrincipalName;
        }
#endif

        public async Task DeleteUserAsync(string mobileNumber)
        {
            if (AdminAadClient == null)
                throw new ApplicationException("Admin aad client is not initialized");
            if (string.IsNullOrWhiteSpace(mobileNumber))
                throw new ArgumentNullException("mobileNumber");

            IUser existUser = await FindUserByMobileNumberAsync(mobileNumber);
            if (existUser == null)
                throw new ApplicationException(string.Format("The user with mobile: {0} has not existed yet", mobileNumber));

            await existUser.DeleteAsync();
        }

#if false // deprecated
        public void DeleteUser(string mobileNumber)
        {
            if (AdminAadClient == null)
                throw new ApplicationException("Admin aad client is not initialized");
            if (string.IsNullOrWhiteSpace(mobileNumber))
                throw new ArgumentNullException("mobileNumber");

            IUser existUser = FindUserByMobileNumber(mobileNumber);
            if (existUser == null)
                throw new ApplicationException(string.Format("The user with mobile: {0} has not existed yet", mobileNumber));

            existUser.DeleteAsync().Wait();
        }
#endif

        private string GetPrincipalNameFromMobile(string mobile)
        {
            return string.Format(PrincipalNameFormat, mobile, _mAadTenantName); //ccs-mobile@aadtentname

        }

    }
}
