﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.Remoting.Messaging;
using CCSPayments.Model;
using CCSPayments.Utils;

namespace CCSPayments
{
    public class WxPayment : Payment, IPayment
    {
        /// <summary>
        /// Download the payment records with specified bill type on a particular day
        /// </summary>
        /// <param name="billDate">format: YYYYMMDD</param>
        /// <param name="billType"></param>
        /// <returns>the payment records on the specified day</returns>
        public override IPaymentData DownloadBill(DateTime billDate, BillType billType)
        {
            string url = "https://api.mch.weixin.qq.com/pay/downloadbill";
            WxConfiguration wxConfig = new WxConfiguration();
            WxDownloadBillRequest inputObj = new WxDownloadBillRequest(wxConfig);
            inputObj.SetBillDate(billDate);
            inputObj.SetBillType(billType);//账单类型
            inputObj.SetSign();//签名
            string xml = inputObj.ToXml();

            Log.Debug("WxPayApi", "DownloadBill request : " + xml);
            HttpService httpClient = new HttpService(wxConfig);
            string response = httpClient.Post(xml, url, false);//调用HTTP通信接口以提交数据到API
            Log.Debug("WxPayApi", "DownloadBill result : " + response);

            WxDownloadBillResponse result = new WxDownloadBillResponse(wxConfig);
            //若接口调用失败会返回xml格式的结果
            if (response.Substring(0, 5) == "<xml>")
            {
                result.FromXml(response);
                result.RequestStatus = false;
                result.ErrorCode = result.GetValue("return_code").ToString();
                result.ErrorMessage = result.GetValue("return_msg").ToString();
            }
            //接口调用成功则返回非xml格式的数据
            else
            {
                result.RequestStatus = true;
                //save the non-structed data for debuging purpose
                result.SetValue(Configuration.ResultKey, response);
                switch (billType)
                {
                    case BillType.ALL:
                    case BillType.REFUND:
                    case BillType.SUCCESS:
                        ParseWxDownloadBillResponseStr(response, billType, result);
                        break;
                    case BillType.REVOKED:  //TODO just save the non-structed data as of now, as no document for this yet
                        break;

                }
            }

           return result;
        }


        private static void ParseWxDownloadBillResponseStr(string responseStr, BillType billType, WxDownloadBillResponse responseObj)
        {
            if(string.IsNullOrWhiteSpace(responseStr))
                throw new ArgumentNullException("responseStr");
            if(responseObj == null)
                throw new ArgumentNullException("responseObj");

            StringReader strReader = new StringReader(responseStr);
            IList<string> lines = new List<string>();

            IList<WxDownloadBillResponseDetails> transRecords = new List<WxDownloadBillResponseDetails>();
            WxDownloadBillResponseSummary responseSummary = null;
            //obtain table head
            string aLine = null;

            do
            {
                aLine = strReader.ReadLine();
                if (aLine != null)
                {
                    lines.Add(aLine.Trim());
                }
            } while (aLine != null);

            int numOfLines = lines.Count;
            if (numOfLines < 1)
            {
                throw new WxPaymentException("Missing Table head");
            }
//            string tableHead = lines[0].Trim();
//            string[] items = tableHead.Split(',');
//            itemList = new List<string[]>();
//            itemList.Add(items);
//
            if (numOfLines < 4)
            {
                throw new WxPaymentException("Must have at least 4 lines: table head, data, summary head, summary data");
            }

            //skip line 0, which is table head
            int lineNum = 1;
            while (lineNum < numOfLines - 2)
            {
                //data details for each transaction
                string transDetail = lines[lineNum++].Trim();
                WxDownloadBillResponseDetails aRecord = ParseATransactionRecordStr(transDetail, billType);
                transRecords.Add(aRecord);
            }

            //string summaryHead = lines[numOfLines - 2].Trim(); //the second last line
            string summaryData = lines[numOfLines - 1].Trim(); //the last line
            responseSummary = ParseResponseSummaryStr(summaryData);


            responseObj.ResponseDetails = transRecords;
            responseObj.ResponseSummary = responseSummary;
        }

        private static WxDownloadBillResponseDetails ParseATransactionRecordStr(string aTransRecord, BillType billType)
        {
            string[] items = aTransRecord.Split(',');
            WxDownloadBillResponseDetails responseDetails = null;
            switch (billType)
            {
                case BillType.ALL:
                    responseDetails = new WxDownloadBillResponseDetailsBillTypeAll();
                    if(items.Length != responseDetails.FieldCount)
                        throw new WxPaymentException("Number of fields does not match");
                    //当日所有订单
                    //
                    //交易时间,公众账号ID,商户号,子商户号,设备号,微信订单号,商户订单号,用户标识,交易类型,交易状态,付款银行,货币种类,总金额,代金券或立减优惠金额,微信退款单号,商户退款单号,退款金额,代金券或立减优惠退款金额，退款类型，退款状态,商品名称,商户数据包,手续费,费率
                    responseDetails.TradeTime = items[0].TrimStart('`');
                    responseDetails.WxAppId = items[1].TrimStart('`');
                    responseDetails.MerchantId = items[2].TrimStart('`');
                    responseDetails.SubMerchantId = items[3].TrimStart('`');
                    responseDetails.DeviceId = items[4].TrimStart('`');
                    responseDetails.WxOrderId = items[5].TrimStart('`');
                    responseDetails.MerchantOrderId = items[6].TrimStart('`');
                    responseDetails.UserId = items[7].TrimStart('`');
                    responseDetails.TransactionType = items[8].TrimStart('`');
                    responseDetails.TransactionStatus = items[9].TrimStart('`');
                    responseDetails.PayerBank = items[10].TrimStart('`');
                    responseDetails.CurrencyType = items[11].TrimStart('`');
                    responseDetails.TotalPaymentAmount = Double.Parse(items[12].TrimStart('`'));
                    responseDetails.TotalCouponPaymentAmount = Double.Parse(items[13].TrimStart('`'));

                    var allDetails = (WxDownloadBillResponseDetailsBillTypeAll) responseDetails;
                    allDetails.WxRefundId = items[14].TrimStart('`');
                    allDetails.MerchantRefundId = items[15].TrimStart('`');
                    allDetails.RefundAmount = Double.Parse(items[16].TrimStart('`'));
                    allDetails.RefundCouponAmount = Double.Parse(items[17].TrimStart('`'));
                    allDetails.RefundType = items[18].TrimStart('`');
                    allDetails.RefundStatus = items[19].TrimStart('`');

                    responseDetails.ProductName = items[20].TrimStart('`');
                    responseDetails.ProductDescription = items[21].TrimStart('`');
                    responseDetails.ProcessingFee = Double.Parse(items[22].TrimStart('`'));
                    responseDetails.ProcessingFeeRate = items[23].TrimStart('`');

                    break;
                case BillType.SUCCESS:
                    responseDetails = new WxDownloadBillResponseDetailsBillTypeSucceed();
                    if (items.Length != responseDetails.FieldCount)
                        throw new WxPaymentException("Number of fields does not match");
                    //TODO
                    break;
                case BillType.REFUND:
                    responseDetails = new WxDownloadBillResponseDetailsBillTypeRefund();
                    if (items.Length != responseDetails.FieldCount)
                        throw new WxPaymentException("Number of fields does not match");
                    //TODO
                    break;
                case BillType.REVOKED:
                    throw new WxPaymentException("Unsupported");
            }
            return responseDetails;
        }

        private static WxDownloadBillResponseSummary ParseResponseSummaryStr(string summaryStr)
        {
            string[] items = summaryStr.Split(',');
            WxDownloadBillResponseSummary respSummary = new WxDownloadBillResponseSummary();
            if (items.Length != respSummary.FieldCount)
                throw new WxPaymentException("Number of fields does not match WxDownloadBillResponseSummary");

            //总交易单数，总交易额，总退款金额，总代金券或立减优惠退款金额，手续费总金额 
            respSummary.TotalTradeCount = int.Parse(items[0].TrimStart('`'));
            respSummary.TotalTradeAmount = Double.Parse(items[1].TrimStart('`'));
            respSummary.TotalRefundAmount = Double.Parse(items[2].TrimStart('`'));
            respSummary.TotalCouponRefundAccount = Double.Parse(items[3].TrimStart('`'));
            respSummary.TotalProcessingFee = Double.Parse(items[4].TrimStart('`'));

            return respSummary;
        }

    }
}
