﻿using System;
using System.Linq;
using System.Linq.Expressions;
using System.Collections.Generic;
using Microsoft.WindowsAzure.Storage.Queue;
using Microsoft.WindowsAzure.Storage;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;

namespace CCSPrototypeCommon.Query
{
    [Serializable]
    public abstract class  AbstractFilter
    {

        public Guid BaseSearchID;

        public string SelectedCategory { get; set; }
        public string ModelTypeName { get; set; }
        public string Property { get; set; }

        public string TargetTypeName { get; set; }


        protected abstract Expression BuildFilterExpression(Expression property);

        private Expression BuildFilterExpressionWithNullChecks(
           Expression filterExpression,
           ParameterExpression parameter,
           Expression property,
           string[] remainingPropertyParts)
        {
            property = Expression.Property(property == null ? parameter : property, remainingPropertyParts[0]);

            if (remainingPropertyParts.Length == 1)
            {
                if (!property.Type.IsValueType || property.Type.IsGenericType && property.Type.GetGenericTypeDefinition() == typeof(Nullable<>))
                {
                    var nullCheckExpression = Expression.NotEqual(property, Expression.Constant(null));
                    filterExpression = CombineExpressions(filterExpression, nullCheckExpression);
                }

                

                if (property.Type.IsGenericType && property.Type.GetGenericTypeDefinition() == typeof(Nullable<>))
                {
                    property = Expression.Property(property, "Value");
                }

                Expression searchExpression = null;
                if (property.Type.IsGenericType && typeof(IEnumerable<>).MakeGenericType(property.Type.GetGenericArguments()).IsAssignableFrom(property.Type))
                {
                    parameter = Expression.Parameter(property.Type.GetGenericArguments().First());

                    searchExpression = ApplySearchExpressionToCollection(
                        parameter,
                        property,
                        this.BuildFilterExpression(parameter));
                }
                else
                {
                    searchExpression = this.BuildFilterExpression(property);
                }

                if (searchExpression == null)
                {
                    return null;
                }
                else
                {
                    return CombineExpressions(filterExpression, searchExpression);
                }
            }
            else
            {
                var nullCheckExpression = Expression.NotEqual(property, Expression.Constant(null));
                filterExpression = CombineExpressions(filterExpression, nullCheckExpression);

                if (property.Type.IsGenericType && typeof(IEnumerable<>).MakeGenericType(property.Type.GetGenericArguments()).IsAssignableFrom(property.Type))
                {
                    parameter = Expression.Parameter(property.Type.GetGenericArguments().First());
                    Expression searchExpression = this.BuildFilterExpressionWithNullChecks(
                        null,
                        parameter,
                        null,
                        remainingPropertyParts.Skip(1).ToArray());

                    if (searchExpression == null)
                    {
                        return null;
                    }
                    else
                    {
                        searchExpression = ApplySearchExpressionToCollection(
                            parameter,
                            property,
                            searchExpression);

                        return CombineExpressions(filterExpression, searchExpression);
                    }
                }
                else
                {
                    return this.BuildFilterExpressionWithNullChecks(filterExpression, parameter, property, remainingPropertyParts.Skip(1).ToArray());
                }
            }
        }

        private static Expression ApplySearchExpressionToCollection(ParameterExpression parameter, Expression property, Expression searchExpression)
        {
            if (searchExpression != null)
            {
                var asQueryable = typeof(Queryable).GetMethods()
                    .Where(m => m.Name == "AsQueryable")
                    .Single(m => m.IsGenericMethod)
                    .MakeGenericMethod(property.Type.GetGenericArguments());

                var anyMethod = typeof(Queryable).GetMethods()
                    .Where(m => m.Name == "Any")
                    .Single(m => m.GetParameters().Length == 2)
                    .MakeGenericMethod(property.Type.GetGenericArguments());

                searchExpression = Expression.Call(
                    null,
                    anyMethod,
                    Expression.Call(null, asQueryable, property),
                    Expression.Lambda(searchExpression, parameter));
            }
            return searchExpression;
        }

        private Expression CombineExpressions(Expression first, Expression second)
        {
            if (first == null)
            {
                return second;
            }
            else
            {
                return Expression.AndAlso(first, second);
            }
        }

        public  IQueryable<T> ApplyToQuery<T>(IQueryable<T> query)
        {
            string[] parts = this.Property.Split('.');

            var parameter = Expression.Parameter(typeof(T), "p");

            Expression filterExpression = this.BuildFilterExpressionWithNullChecks(null, parameter, null, parts);

            if (filterExpression == null)
            {
                return query;
            }
            else
            {
                var predicate = Expression.Lambda<Func<T, bool>>(filterExpression, parameter);
                return query.Where(predicate);
            }
        }

    }
}
