﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;

using CCSAdmin;
using CCSPrototypeCommon;
using CCSPrototypeCommon.Query;
using HomeWeb.Models;
using System.Data;
using Microsoft.WindowsAzure.Storage.Blob;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.ServiceRuntime;
using System.Globalization;

namespace HomeWeb.Controllers
{
    public class BaseController : Controller
    {

        protected CloudBlobContainer _vendorProfilePictureBlobContainer;

        public BaseController()
        {
            CloudStorageAccount storageAccount = CloudStorageAccount.Parse(RoleEnvironment.GetConfigurationSettingValue("StorageConnectionString"));
            CloudBlobClient blobClient = storageAccount.CreateCloudBlobClient();

            var blobAccess = new BlobContainerPermissions();
            blobAccess.PublicAccess = BlobContainerPublicAccessType.Blob;

            _vendorProfilePictureBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.VendorProfilePictureBlobContainerName);
            _vendorProfilePictureBlobContainer.CreateIfNotExists();
            _vendorProfilePictureBlobContainer.SetPermissions(blobAccess);


        }
        public static string GetCurrentCustomerID(HttpSessionStateBase Session, System.Security.Principal.IPrincipal User)
        {
            var ob = Session[SessionConstants.CustomerID];
            if (ob != null)
                return ob.ToString();

            string customerID = string.Empty;
            string customerName = string.Empty;
            string companyLogo = string.Empty;
            if (User != null)
            {
                // customerId must exist, unless this is a manually created account (e.g. WaterCompany account)
                // in this case, we hardcode the customerID for this water company account.
                customerID = Constants.DefaultWaterCompanyCustomerID;
                /* Comment out the real code for test purpose
                string userId = User.Identity.Name;

                using (var vendorEFContext = (VendorContext)VendorContextFactory.GetInstance().GetDBEFContext())
                {
                    var record = vendorEFContext.StaffTable.Where(entry => entry.ADAccount == userId).FirstOrDefault();
                    if (record != null)
                    {
                        customerID = record.CustomerID.ToString();
                    }
                }
                */
            }

            if (!String.IsNullOrEmpty(customerID))
            {
                Session[SessionConstants.CustomerID] = customerID;

                if (!String.IsNullOrEmpty(customerName))
                {
                    Session[SessionConstants.CustomerName] = customerName;
                }

                if (!String.IsNullOrEmpty(companyLogo))
                {
                    Session[SessionConstants.CompanyLogo] = companyLogo;
                }
            }
            return customerID;
        }

        // return the corresponding customer ID (aka Company ID) from a logged user
        protected string GetCurrentCustomerID()
        {
            string customerID = GetCurrentCustomerID(Session, User);
            if (string.IsNullOrEmpty(customerID))
            {
                throw new Exception(ErrorConstants.NoAssociatedCompany);
            }
            else
            {
                return customerID;
            }
        }

        protected CCS_Staff_V1 GetCurrentStaff()
        {
            try
            {
                var adAccount = User.Identity.Name;
                using (var vendorEFContext = (VendorContext)VendorContextFactory.GetInstance().GetDBEFContext())
                {
                    var record = vendorEFContext.StaffTable.Where(entry => entry.ADAccount == adAccount).FirstOrDefault();
                    if (record != null)
                    {
                        return record;
                    }
                }
            }
            catch (DataException)
            {
                return null;
            }
            return null;
        }

        protected string GetProfilePictureUri(string staffID)
        {
            string blobPrefix = staffID + '/';
            var blobList = _vendorProfilePictureBlobContainer.ListBlobs(blobPrefix, true);
            var dateTimeFormat = "yyyyMMddHHmmss";
            var uri = "";
            DateTime latestDate = new DateTime();
            foreach (var blob in blobList)
            {
                DateTime uploadDate;
                if (!DateTime.TryParseExact(blob.Parent.Uri.MakeRelativeUri(blob.Uri).ToString(),
                    dateTimeFormat, CultureInfo.InvariantCulture, DateTimeStyles.None, out uploadDate))
                {
                    if (string.IsNullOrWhiteSpace(uri))
                    {
                        uri = blob.Uri.AbsoluteUri;
                    }
                }
                else if (latestDate < uploadDate)
                {
                    latestDate = uploadDate;
                    uri = blob.Uri.AbsoluteUri;
                }
            }
            return uri;
        }

        public static GeoRegion GetRegion(string region)
        {
            if (string.IsNullOrEmpty(region))
            {
                return GeoRegion.Beijing;
            }
            switch (region.ToLower())
            {
                case "北京":
                case "beijing":
                    return GeoRegion.Beijing;
                case "上海":
                case "shanghai":
                    return GeoRegion.Shanghai;
                case "天津":
                case "tianjin":
                    return GeoRegion.Tianjin;
                case "重庆":
                case "chongqing":
                    return GeoRegion.Chongqing;
                case "湖北":
                case "hubei":
                    return GeoRegion.Hubei;
                default:
                    return GeoRegion.Beijing;
            }
        }
    }
}