﻿using CCSAdmin;
using CCSPrototypeCommon;
using HomeWeb.Models;
using Microsoft.WindowsAzure.ServiceRuntime;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.Storage.Blob;
using Microsoft.WindowsAzure.Storage.Queue;
using Microsoft.WindowsAzure.Storage.RetryPolicies;
using Microsoft.WindowsAzure.Storage.Table;
using Newtonsoft.Json;
using PushSharp;
using PushSharp.Apple;
using System;
using System.Collections.Generic;
using System.Data;
using System.Data.Entity.Infrastructure;
using System.Data.Entity.Spatial;
using System.Data.Entity.Validation;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using System.Threading.Tasks;
using System.Web;
using System.Web.Mvc;

namespace HomeWeb.Controllers
{
    // Merchant - Staff (Water Company)
    public partial class MerchantController : QueryableController
    {
        static private CloudBlobContainer _adsBlobContainer;
        static private CloudBlobContainer _rawDataBlobContainer;
        static private CloudBlobContainer _adsMediaBlobContainer;
        static private CloudBlobContainer _adsBlobArchiveContainer;

        private CloudBlobContainer _billBlobContainer;

        private CloudQueue _billQueue;

        static private CloudQueue _rawDataQueue;
        static private CloudQueue filesQueue;
        private static CloudBlobContainer filesBlobContainer;
        private static CloudBlobContainer paymentsBlobContainer;
        private static Dictionary<string, Uri> BlobHashMap = new Dictionary<string, Uri>();

        private CloudTable _billTemplateTable;

        const string QueryDownloadBlobName = "DownloadQueryResult";

        private static BillContext _billDB = (BillContext)BillContextFactory.GetInstance().GetDBEFContext();

        public MerchantController() : base()
        {
            InitializeCloudStorage();
        }

        public CCS_Customer_V1 GetCustomerInfo(Guid customerID)
        {
            var customerEFContext = (CustomerContext)CustomerContextFactory.GetInstance().GetDBEFContext();
            var record = customerEFContext.CustomerTable.Where(entry => entry.CustomerID == customerID).FirstOrDefault();
            if (record != null)
            {
                return record;
            }

            throw new Exception(ErrorConstants.NoAssociatedCompany);
        }


        protected override void InitializeCloudStorage()
        {

            // Open storage account
            CloudStorageAccount storageAccount = CloudStorageAccount.Parse(RoleEnvironment.GetConfigurationSettingValue("StorageConnectionString"));
            CloudBlobClient blobClient = storageAccount.CreateCloudBlobClient();
            blobClient.DefaultRequestOptions.RetryPolicy = new LinearRetry(TimeSpan.FromSeconds(3), 3);

            // Init Ads text container
            _adsBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.AdsBlobContainerName);
            //_adsBlobContainer.CreateIfNotExists();
            _adsBlobArchiveContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.AdsBlobArchiveContainerName);
            //_adsBlobArchiveContainer.CreateIfNotExists();

            // Init Ads multimedia container
            _adsMediaBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.AdsMultimediaBlobContainerName);
            //_adsMediaBlobContainer.CreateIfNotExists();
            var blobAccess = new BlobContainerPermissions();
            blobAccess.PublicAccess = BlobContainerPublicAccessType.Blob;
            _adsMediaBlobContainer.SetPermissions(blobAccess);

            filesBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.BillStorageName);

            //filesBlobContainer.CreateIfNotExists();

            paymentsBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.PaymentStorageName);

            //paymentsBlobContainer.CreateIfNotExists();

            // Get queue context object
            CloudQueueClient queueClient = storageAccount.CreateCloudQueueClient();
            queueClient.DefaultRequestOptions.RetryPolicy = new LinearRetry(TimeSpan.FromSeconds(3), 3);
            filesQueue = queueClient.GetQueueReference(CCSPrototypeCommon.Constants.BillStorageName);

            //filesQueue.CreateIfNotExists();


            blobClient.DefaultRequestOptions.RetryPolicy = new LinearRetry(TimeSpan.FromSeconds(3), 3);
            _billBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.BillStorageName);
            //_billBlobContainer.CreateIfNotExists();
            _rawDataBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.RawDataStorageName);
            //_rawDataBlobContainer.CreateIfNotExists();
            queryBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.QueryStorageName);
            //queryBlobContainer.CreateIfNotExists();

            _billQueue = queueClient.GetQueueReference(CCSPrototypeCommon.Constants.BillStorageName);
            //_billQueue.CreateIfNotExists();
            queryInitiateQueue = queueClient.GetQueueReference(CCSPrototypeCommon.Constants.QueryInitStorageName);
            //queryInitiateQueue.CreateIfNotExists();
            queryDownloadQueue = queueClient.GetQueueReference(CCSPrototypeCommon.Constants.QueryDownloadStorageName);
            //queryDownloadQueue.CreateIfNotExists();

            _rawDataQueue = queueClient.GetQueueReference(CCSPrototypeCommon.Constants.RawDataStorageName);
            //_rawDataQueue.CreateIfNotExists();

            CloudTableClient tableClient = storageAccount.CreateCloudTableClient();
            tableClient.DefaultRequestOptions.RetryPolicy = new LinearRetry(TimeSpan.FromSeconds(3), 3);
            _billTemplateTable = tableClient.GetTableReference(CCSPrototypeCommon.Constants.DataTemplateTableName);
            //_billTemplateTable.CreateIfNotExists();

        }

        // Show default page
        public ActionResult Index()
        {
            return View();
        }

        [Authorize(Roles = "Staff, Admin, SysAdmin")]
        public ActionResult RegisterVendor(string companyIDstr)
        {
            List<string> roleList = new List<string>();
            if (User.IsInRole(OperationLevel.Admin.ToString()))
            {
                roleList.Add(OperationLevel.Staff.GetDisplayName());
            }
            if (User.IsInRole(OperationLevel.SysAdmin.ToString()))
            {
                roleList.Add(OperationLevel.Admin.GetDisplayName());
            }
            if (User.IsInRole(OperationLevel.Staff.ToString()))
            {
                roleList.Add(OperationLevel.Vendor.GetDisplayName());
                roleList.Add(OperationLevel.Gatherer.GetDisplayName());
                roleList.Add(OperationLevel.MeterReader.GetDisplayName());
            }
            if (!string.IsNullOrEmpty(companyIDstr))
            {
                ViewBag.CompanyID = companyIDstr;
            }
            ViewBag.RoleList = new SelectList(roleList);
            Session["companyIDstr"] = companyIDstr;
            return View();
        }

        [Authorize(Roles = "Staff, Admin")]
        public ActionResult ManageVendor()
        {
            var curCustomerID = new Guid(GetCurrentCustomerID());
            var vendorMap = new Dictionary<Guid, int>();
            var vendorList = new List<VendorModel>();
            try
            {
                using (var vendorEFContext = (VendorContext)VendorContextFactory.GetInstance().GetDBEFContext())
                {
                    var recordQuery = (from staff in vendorEFContext.StaffTable.AsNoTracking().Where(s => (s.CustomerID == curCustomerID && s.State == CustomerAccountState.Active))
                                       join staffInfo in vendorEFContext.StaffInfoTable.AsNoTracking().Where(si => si.Status == DBItemState.Valid).DefaultIfEmpty()
                                       on staff.StaffID equals staffInfo.StaffID
                                       into ssi
                                       from item in ssi.DefaultIfEmpty()
                                       select new
                                       {
                                           Staff = staff,
                                           StaffInfo = item
                                       });
                    if (User.IsInRole(OperationLevel.Staff.ToString()))
                    {
                        recordQuery = recordQuery.Where(r => (r.Staff.Role == OperationLevel.Vendor ||
                            r.Staff.Role == OperationLevel.Gatherer ||
                            r.Staff.Role == OperationLevel.MeterReader));
                    }
                    else if (User.IsInRole(OperationLevel.Admin.ToString()))
                    {
                        recordQuery = recordQuery.Where(r => r.Staff.Role == OperationLevel.Staff);
                    }
                    var recordList = recordQuery.ToList();
                    foreach (var record in recordList)
                    {
                        if (vendorMap.ContainsKey(record.Staff.StaffID))
                        {
                            if (record.StaffInfo != null)
                            {
                                vendorList[vendorMap[record.Staff.StaffID]].StaffInfoList.Add(record.StaffInfo);
                            }
                        }
                        else
                        {
                            vendorMap.Add(record.Staff.StaffID, vendorList.Count);
                            var vendor = new VendorModel()
                            {
                                StaffTable = record.Staff
                            };
                            if (record.StaffInfo != null)
                            {
                                vendor.StaffInfoList.Add(record.StaffInfo);
                            }
                            vendorList.Add(vendor);
                        }
                    }
                }
            }
            catch (DataException)
            {
                throw;
            }
            return View(vendorList);
        }

        [Authorize(Roles = "Staff, Admin, SysAdmin")]
        public async Task<ActionResult> DisableVendor(Guid? staffID)
        {
            string nextAction = "ManageVendor";
            if (User.IsInRole(OperationLevel.SysAdmin.ToString()))
            {
                nextAction = "SetCompanyAdmin";
            }

            if (staffID == null)
            {
                return RedirectToAction(nextAction);
            }
            try
            {
                using (var vendorEFContext = (VendorContext)VendorContextFactory.GetInstance().GetDBEFContext())
                {
                    var vendor = new CCS_Staff_V1()
                    {
                        StaffID = (Guid)staffID,
                        State = CustomerAccountState.Disable
                    };
                    vendorEFContext.StaffTable.Attach(vendor);
                    vendorEFContext.Configuration.ValidateOnSaveEnabled = false;
                    vendorEFContext.Entry(vendor).Property(c => c.State).IsModified = true;
                    await vendorEFContext.SaveChangesAsync();
                }
                await CcsAdmin.Instance.DisableUserAsync(staffID.ToString());
            }
            catch (DbEntityValidationException e)
            {
                foreach (var eve in e.EntityValidationErrors)
                {

                }
            }
            catch (DbUpdateException)
            {
                throw;
            }
            catch (RetryLimitExceededException)
            {
                throw;
            }
            catch (Exception)
            {
                throw;
            }
            return RedirectToAction(nextAction);
        }

        [Authorize(Roles = "Staff, Admin, SysAdmin")]
        public ActionResult EditVendorInfo(Guid? staffID)
        {
            string nextAction = "ManageVendor";
            if (User.IsInRole(OperationLevel.SysAdmin.ToString()))
            {
                nextAction = "SetCompanyAdmin";
            }
            if (staffID == null)
            {
                return RedirectToAction(nextAction);
            }
            try
            {
                using (var vendorEFContext = (VendorContext)VendorContextFactory.GetInstance().GetDBEFContext())
                {
                    var record = (from staff in vendorEFContext.StaffTable.AsNoTracking().Where(s => (s.StaffID == staffID && s.State == CustomerAccountState.Active))
                                  join staffInfo in vendorEFContext.StaffInfoTable.AsNoTracking().Where(si => si.Status == DBItemState.Valid).DefaultIfEmpty()
                                  on staff.StaffID equals staffInfo.StaffID
                                  into ssi
                                  from item in ssi.DefaultIfEmpty()
                                  select new
                                  {
                                      Staff = staff,
                                      StaffInfo = item
                                  }).ToList();
                    var vendor = new VendorModel()
                    {
                        StaffTable = record[0].Staff
                    };
                    foreach (var info in record)
                    {
                        if (info.StaffInfo != null)
                        {
                            vendor.StaffInfoList.Add(info.StaffInfo);
                        }
                    }
                    vendor.ProfilePictureUri = GetProfilePictureUri(staffID.ToString());
                    VendorModel.Current = vendor;
                }
            }
            catch (DataException)
            {
                throw;
            }
            return View(VendorModel.Current);
        }

        [HttpPost]
        //[Authorize(Roles = "Staff, Admin, SysAdmin")]
        public ActionResult ViewVendorInfo(Guid? StaffGUID)
        {
            if (StaffGUID == null)
            {
                return PartialView("_ViewVendorInfo");
            }
            CCS_Staff_V1 StaffRecord = null;
            using (var vendorEFContext = (VendorContext)VendorContextFactory.GetInstance().GetDBEFContext())
            {
                vendorEFContext.Configuration.LazyLoadingEnabled = false;
                StaffRecord = vendorEFContext.StaffTable.Find(StaffGUID);

                if (vendorEFContext.Entry(StaffRecord).Collection(s => s.StaffInfos).IsLoaded == false)
                {
                    vendorEFContext.Entry(StaffRecord).Collection(s => s.StaffInfos).Load();
                }


                if (StaffRecord.StaffInfos == null)
                {
                    StaffRecord.StaffInfos = new List<CCS_StaffInfo_V1>();
                }
                else if (StaffRecord.StaffInfos.Count > 0)
                {
                    int count = StaffRecord.StaffInfos.Count;
                    List<CCS_StaffInfo_V1> DisabledInfo = new List<CCS_StaffInfo_V1>();
                    foreach (CCS_StaffInfo_V1 staffInfo in StaffRecord.StaffInfos)
                    {
                        if (staffInfo.Status == DBItemState.Invalid)
                        {
                            DisabledInfo.Add(staffInfo);
                        }
                    }

                    foreach (CCS_StaffInfo_V1 staffInfo in DisabledInfo)
                    {
                        StaffRecord.StaffInfos.Remove(staffInfo);
                    }
                }
            }

            return PartialView("_ViewVendorInfo", StaffRecord); 
        }

        [HttpPost]
        [Authorize(Roles = "Staff, Admin, SysAdmin")]
        public async Task<ActionResult> EditVendorInfo(FormCollection collection,
            [Bind(Include = "StaffTable")]VendorModel vendor)
        {
            string nextAction = "ManageVendor";
            if (User.IsInRole(OperationLevel.SysAdmin.ToString()))
            {
                nextAction = "SetCompanyAdmin";
            }
            try
            {
                if (ModelState.IsValid)
                {
                    using (var vendorEFContext = (VendorContext)VendorContextFactory.GetInstance().GetDBEFContext())
                    {
                        var unmodifiedVendor = VendorModel.Current;
                        vendor.StaffTable.LastUpdatedDate = DateTime.UtcNow;
                        vendor.StaffTable.StaffID = unmodifiedVendor.StaffTable.StaffID;
                        vendor.StaffTable.ADAccount = unmodifiedVendor.StaffTable.ADAccount;
                        vendor.StaffTable.ADDomain = unmodifiedVendor.StaffTable.ADDomain;
                        vendor.StaffTable.CustomerID = unmodifiedVendor.StaffTable.CustomerID;
                        vendor.StaffTable.Role = unmodifiedVendor.StaffTable.Role;
                        vendor.StaffTable.State = unmodifiedVendor.StaffTable.State;
                        vendor.StaffTable.geography = DbGeography.FromText(string.Format("POINT({1} {0})", vendor.StaffTable.latitude, vendor.StaffTable.longitude));
                        vendorEFContext.Entry(vendor.StaffTable).State = System.Data.Entity.EntityState.Modified;
                        if (collection.AllKeys.Contains("item.Name") &&
                            collection.AllKeys.Contains("item.Value"))
                        {
                            string[] keyList = collection["item.Name"].Split(',');
                            string[] valueList = collection["item.Value"].Split(',');
                            var newLength = keyList.Length;
                            var oldLength = unmodifiedVendor.StaffInfoList.Count;
                            for (int index = 0; index < ((newLength <= oldLength) ? newLength : oldLength); ++index)
                            {
                                if (string.IsNullOrWhiteSpace(keyList[index]))
                                {
                                    continue;
                                }
                                var staffInfo = new CCS_StaffInfo_V1()
                                {
                                    StaffInfoID = unmodifiedVendor.StaffInfoList[index].StaffInfoID,
                                    StaffID = unmodifiedVendor.StaffTable.StaffID,
                                    Name = keyList[index],
                                    Value = valueList[index],
                                    Status = unmodifiedVendor.StaffInfoList[index].Status
                                };
                                vendorEFContext.Entry(staffInfo).State = System.Data.Entity.EntityState.Modified;
                            }
                            for (int index = oldLength; index < newLength; ++index)
                            {
                                if (string.IsNullOrWhiteSpace(keyList[index]))
                                {
                                    continue;
                                }
                                var staffInfo = new CCS_StaffInfo_V1()
                                {
                                    StaffID = unmodifiedVendor.StaffTable.StaffID,
                                    Name = keyList[index],
                                    Value = valueList[index]
                                };
                                vendorEFContext.Entry(staffInfo).State = System.Data.Entity.EntityState.Added;
                            }
                            for (int index = newLength; index < oldLength; ++index)
                            {
                                var staffInfo = new CCS_StaffInfo_V1()
                                {
                                    StaffInfoID = unmodifiedVendor.StaffInfoList[index].StaffInfoID
                                };
                                vendorEFContext.Entry(staffInfo).State = System.Data.Entity.EntityState.Deleted;
                            }
                        }
                        await vendorEFContext.SaveChangesAsync();
                    }
                }
            }
            catch (DbEntityValidationException e)
            {
                foreach (var eve in e.EntityValidationErrors)
                {

                }
            }
            catch (DbUpdateException)
            {
                throw;
            }
            return RedirectToAction(nextAction);
        }

        [HttpPost]
        [Authorize(Roles = "Staff, Admin, SysAdmin")]
        public async Task<ActionResult> RegisterVendor(FormCollection collection,
            string roleType, string companyIDstr,
            [Bind(Include = "StaffTable, Password, ComparePassword")]CustomerModel customer)
        {
            // for vendor register
            try
            {
                Guid curCustomerID;
                if (string.IsNullOrEmpty(companyIDstr) || !Guid.TryParse(companyIDstr, out curCustomerID))
                {
                    if (User.IsInRole(OperationLevel.SysAdmin.ToString()))
                    {
                        return RedirectToAction("SetCompanyAdmin");
                    }
                    curCustomerID = new Guid(GetCurrentCustomerID());
                }
                var company = GetCustomerInfo(curCustomerID);
                if (ModelState.IsValid)
                {
                    // Fill default values
                    customer.StaffTable.LastUpdatedDate = DateTime.UtcNow;

                    // To do: Convert to phonetic symbol
                    customer.StaffTable.CustomerID = curCustomerID;

                    using (var vendorEFContext = (VendorContext)VendorContextFactory.GetInstance().GetDBEFContext())
                    {
                        if (string.IsNullOrWhiteSpace(customer.StaffTable.Name) ||
                            string.IsNullOrWhiteSpace(customer.Password) ||
                            string.IsNullOrWhiteSpace(customer.ComparePassword))
                        {
                            // to do return webpage an error
                            return View(customer);
                        }
                        customer.StaffTable.geography = DbGeography.FromText(string.Format("POINT({1} {0})", customer.StaffTable.latitude, customer.StaffTable.longitude));
                        vendorEFContext.StaffTable.Add(customer.StaffTable);

                        if (collection.AllKeys.Contains("item.Name") &&
                            collection.AllKeys.Contains("item.Value"))
                        {
                            string[] keyList = collection["item.Name"].Split(',');
                            string[] valueList = collection["item.Value"].Split(',');
                            for (int index = 0; index < keyList.Length; ++index)
                            {
                                if (string.IsNullOrWhiteSpace(keyList[index]))
                                {
                                    continue;
                                }
                                var staffInfo = new CCS_StaffInfo_V1()
                                {
                                    Name = keyList[index],
                                    Value = valueList[index]
                                };
                                vendorEFContext.StaffInfoTable.Add(staffInfo);
                            }
                        }
                        if (string.Compare(roleType, OperationLevel.Staff.GetDisplayName(), StringComparison.OrdinalIgnoreCase) == 0)
                        {
                            customer.StaffTable.Role = OperationLevel.Staff;
                        }
                        else if (string.Compare(roleType, OperationLevel.Admin.GetDisplayName(), StringComparison.OrdinalIgnoreCase) == 0)
                        {
                            customer.StaffTable.Role = OperationLevel.Admin;
                        }
                        else if (string.Compare(roleType, OperationLevel.Gatherer.GetDisplayName(), StringComparison.OrdinalIgnoreCase) == 0)
                        {
                            customer.StaffTable.Role = OperationLevel.Gatherer;
                        }
                        else if (string.Compare(roleType, OperationLevel.MeterReader.GetDisplayName(), StringComparison.OrdinalIgnoreCase) == 0)
                        {
                            customer.StaffTable.Role = OperationLevel.MeterReader;
                        }
                        else
                        {
                            customer.StaffTable.Role = OperationLevel.Vendor;
                        }
                        customer.StaffTable.ADDomain = company.DomainName;
                        customer.StaffTable.ADAccount =
                            await CcsAdmin.Instance.CreateUserAsync(customer.StaffTable.Cellphone.ToString(),
                            customer.Password, null, customer.StaffTable.Role.ToString(), company.DomainName);
                        vendorEFContext.SaveChanges();

                        return View("RegisterResult", new MerchantRegisterUserModel { VendorAccountName = customer.StaffTable.ADAccount });
                    }
                }
            }
            catch (DbEntityValidationException e)
            {
                foreach (var eve in e.EntityValidationErrors)
                {

                }
            }
            catch (DbUpdateException)
            {
                throw;
            }
            catch (RetryLimitExceededException /* dex */)
            {
                //Log the error (uncomment dex variable name and add a line here to write a log.)
                ModelState.AddModelError("", "Unable to save changes. Try again, and if the problem persists, see your system administrator.");
            }
            List<string> roleList = new List<string>();
            if (!User.IsInRole(OperationLevel.Staff.ToString()))
            {
                roleList.Add(OperationLevel.Staff.GetDisplayName());
            }
            if (User.IsInRole(OperationLevel.SysAdmin.ToString()))
            {
                roleList.Add(OperationLevel.Admin.GetDisplayName());
            }
            ViewBag.RoleList = new SelectList(roleList);
            return View(customer);
        }

        [Authorize(Roles = "SysAdmin")]
        public ActionResult RegisterCompany()
        {
            var selectList = Enum.GetValues(
                typeof(CCSPrototypeCommon.GeoRegion)).Cast<CCSPrototypeCommon.GeoRegion>().Where(
                e => e != CCSPrototypeCommon.GeoRegion.Unkown).Select(
                e => new SelectListItem
                {
                    Value = ((int)e).ToString(),
                    Text = e.GetDisplayName()
                });
            ViewBag.RegionList = selectList;
            return View();
        }

        [HttpPost]
        [Authorize(Roles = "SysAdmin")]
        public ActionResult RegisterCompany(FormCollection collection,
            [Bind(Include = "CustomerTable")]CompanyModel company)
        {
            try
            {
                if (ModelState.IsValid)
                {
                    // Fill default values
                    company.CustomerTable.LastUpdatedDate = DateTime.UtcNow;

                    using (var customerEFContext = (CustomerContext)CustomerContextFactory.GetInstance().GetDBEFContext())
                    {
                        if (string.IsNullOrWhiteSpace(company.CustomerTable.Name) ||
                            string.IsNullOrWhiteSpace(company.CustomerTable.DomainName))
                        {
                            // to do return webpage an error
                            return RedirectToAction("RegisterCompany");
                        }
                        customerEFContext.CustomerTable.Add(company.CustomerTable);

                        if (collection.AllKeys.Contains("item.Name") &&
                            collection.AllKeys.Contains("item.Value"))
                        {
                            string[] keyList = collection["item.Name"].Split(',');
                            string[] valueList = collection["item.Value"].Split(',');
                            for (int index = 0; index < keyList.Length; ++index)
                            {
                                if (string.IsNullOrWhiteSpace(keyList[index]))
                                {
                                    continue;
                                }
                                var customerInfo = new CCS_CustomerInfo_V1()
                                {
                                    Name = keyList[index],
                                    Value = valueList[index]
                                };
                                customerEFContext.CustomerInfoTable.Add(customerInfo);
                            }
                        }
                        customerEFContext.SaveChanges();
                        return RedirectToAction("ManageCompany");
                    }
                }
            }
            catch (DbEntityValidationException e)
            {
                foreach (var eve in e.EntityValidationErrors)
                {

                }
            }
            catch (DbUpdateException)
            {
                throw;
            }
            catch (RetryLimitExceededException /* dex */)
            {
                //Log the error (uncomment dex variable name and add a line here to write a log.)
                ModelState.AddModelError("", "Unable to save changes. Try again, and if the problem persists, see your system administrator.");
            }
            var selectList = Enum.GetValues(
                typeof(CCSPrototypeCommon.GeoRegion)).Cast<CCSPrototypeCommon.GeoRegion>().Where(
                e => e != CCSPrototypeCommon.GeoRegion.Unkown).Select(
                e => new SelectListItem
                {
                    Value = ((int)e).ToString(),
                    Text = e.GetDisplayName()
                });
            ViewBag.RegionList = selectList;
            return View();
        }

        [Authorize(Roles = "SysAdmin")]
        public ActionResult ManageCompany()
        {
            var companyInfoMap = new Dictionary<Guid, int>();
            var companyList = new List<CompanyModel>();
            try
            {
                using (var customerEFContext = (CustomerContext)CustomerContextFactory.GetInstance().GetDBEFContext())
                {
                    var recordList = (from company in customerEFContext.CustomerTable.AsNoTracking().Where(c => (c.State == CustomerAccountState.Active))
                                      join companyInfo in customerEFContext.CustomerInfoTable.AsNoTracking().Where(ci => ci.Status == DBItemState.Valid).DefaultIfEmpty()
                                      on company.CustomerID equals companyInfo.CustomerID
                                      into ssi
                                      from item in ssi.DefaultIfEmpty()
                                      select new
                                      {
                                          Company = company,
                                          CompanyInfo = item
                                      }).ToList();
                    foreach (var record in recordList)
                    {
                        if (companyInfoMap.ContainsKey(record.Company.CustomerID))
                        {
                            if (record.CompanyInfo != null)
                            {
                                companyList[companyInfoMap[record.Company.CustomerID]].CustomerInfoList.Add(record.CompanyInfo);
                            }
                        }
                        else
                        {
                            companyInfoMap.Add(record.Company.CustomerID, companyList.Count);
                            var company = new CompanyModel()
                            {
                                CustomerTable = record.Company
                            };
                            if (record.CompanyInfo != null)
                            {
                                company.CustomerInfoList.Add(record.CompanyInfo);
                            }
                            companyList.Add(company);
                        }
                    }
                }
            }
            catch (DataException)
            {
                throw;
            }
            return View(companyList);
        }

        [Authorize(Roles = "SysAdmin")]
        public ActionResult EditCompanyInfo(Guid? companyID)
        {
            if (companyID == null)
            {
                return RedirectToAction("ManageCompany");
            }
            try
            {
                using (var customerEFContext = (CustomerContext)CustomerContextFactory.GetInstance().GetDBEFContext())
                {
                    var record = (from company in customerEFContext.CustomerTable.AsNoTracking().Where(c => (c.CustomerID == companyID && c.State == CustomerAccountState.Active))
                                  join companyInfo in customerEFContext.CustomerInfoTable.AsNoTracking().Where(ci => ci.Status == DBItemState.Valid).DefaultIfEmpty()
                                  on company.CustomerID equals companyInfo.CustomerID
                                  into ssi
                                  from item in ssi.DefaultIfEmpty()
                                  select new
                                  {
                                      Company = company,
                                      CompanyInfo = item
                                  }).ToList();
                    var companay = new CompanyModel()
                    {
                        CustomerTable = record[0].Company
                    };
                    foreach (var info in record)
                    {
                        if (info.CompanyInfo != null)
                        {
                            companay.CustomerInfoList.Add(info.CompanyInfo);
                        }
                    }
                    CompanyModel.Current = companay;
                }
            }
            catch (DataException)
            {
                throw;
            }
            var selectList = Enum.GetValues(
                typeof(CCSPrototypeCommon.GeoRegion)).Cast<CCSPrototypeCommon.GeoRegion>().Where(
                e => e != CCSPrototypeCommon.GeoRegion.Unkown).Select(
                e => new SelectListItem
                {
                    Value = ((int)e).ToString(),
                    Text = e.GetDisplayName()
                });
            ViewBag.RegionList = selectList;

            return View(CompanyModel.Current);
        }


        [HttpPost]
        [Authorize(Roles = "SysAdmin")]
        public async Task<ActionResult> EditCompanyInfo(FormCollection collection,
            [Bind(Include = "CustomerTable")]CustomerModel company)
        {
            try
            {
                if (ModelState.IsValid)
                {
                    using (var customerEFContext = (CustomerContext)CustomerContextFactory.GetInstance().GetDBEFContext())
                    {
                        var unmodifiedCompany = CompanyModel.Current;
                        company.CustomerTable.LastUpdatedDate = DateTime.UtcNow;
                        company.CustomerTable.CustomerID = unmodifiedCompany.CustomerTable.CustomerID;
                        company.CustomerTable.State = unmodifiedCompany.CustomerTable.State;
                        customerEFContext.Entry(company.CustomerTable).State = System.Data.Entity.EntityState.Modified;
                        if (collection.AllKeys.Contains("item.Name") &&
                            collection.AllKeys.Contains("item.Value"))
                        {
                            string[] keyList = collection["item.Name"].Split(',');
                            string[] valueList = collection["item.Value"].Split(',');
                            var newLength = keyList.Length;
                            var oldLength = unmodifiedCompany.CustomerInfoList.Count;
                            for (int index = 0; index < ((newLength <= oldLength) ? newLength : oldLength); ++index)
                            {
                                if (string.IsNullOrWhiteSpace(keyList[index]))
                                {
                                    continue;
                                }
                                var customerInfo = new CCS_CustomerInfo_V1()
                                {
                                    CustomerInfoID = unmodifiedCompany.CustomerInfoList[index].CustomerInfoID,
                                    CustomerID = unmodifiedCompany.CustomerTable.CustomerID,
                                    Name = keyList[index],
                                    Value = valueList[index],
                                    Status = unmodifiedCompany.CustomerInfoList[index].Status
                                };
                                customerEFContext.Entry(customerInfo).State = System.Data.Entity.EntityState.Modified;
                            }
                            for (int index = oldLength; index < newLength; ++index)
                            {
                                if (string.IsNullOrWhiteSpace(keyList[index]))
                                {
                                    continue;
                                }
                                var customerInfo = new CCS_CustomerInfo_V1()
                                {
                                    CustomerID = unmodifiedCompany.CustomerTable.CustomerID,
                                    Name = keyList[index],
                                    Value = valueList[index]
                                };
                                customerEFContext.Entry(customerInfo).State = System.Data.Entity.EntityState.Added;
                            }
                            for (int index = newLength; index < oldLength; ++index)
                            {
                                var staffInfo = new CCS_StaffInfo_V1()
                                {
                                    StaffInfoID = unmodifiedCompany.CustomerInfoList[index].CustomerInfoID
                                };
                                customerEFContext.Entry(staffInfo).State = System.Data.Entity.EntityState.Deleted;
                            }
                        }
                        await customerEFContext.SaveChangesAsync();
                    }
                }
            }
            catch (DbEntityValidationException e)
            {
                foreach (var eve in e.EntityValidationErrors)
                {

                }
            }
            catch (DbUpdateException)
            {
                throw;
            }
            return RedirectToAction("ManageCompany");
        }


        [Authorize(Roles = "SysAdmin")]
        public ActionResult SetCompanyAdmin()
        {
            var companyList = new List<CompanyModel>();
            try
            {
                using (var customerEFContext = (CustomerContext)CustomerContextFactory.GetInstance().GetDBEFContext())
                {
                    var recordList = (from company in customerEFContext.CustomerTable.AsNoTracking().Where(c => (c.State == CustomerAccountState.Active))
                                      select new
                                      {
                                          Company = company,
                                      }).ToList();
                    foreach (var record in recordList)
                    {
                        var company = new CompanyModel()
                            {
                                CustomerTable = record.Company
                            };
                        companyList.Add(company);
                    }
                }
            }
            catch (DataException)
            {
                throw;
            }
            return View(companyList);
        }

        [HttpPost]
        [Authorize(Roles = "SysAdmin")]
        public ActionResult OnSelectCompanyChanged(string companyIDstr)
        {
            Guid companyID;
            if (string.IsNullOrEmpty(companyIDstr) || !Guid.TryParse(companyIDstr, out companyID))
            {
                return RedirectToAction("SetCompanyAdmin");
            }
            var vendorMap = new Dictionary<Guid, int>();
            var vendorList = new List<VendorModel>();
            try
            {
                using (var vendorEFContext = (VendorContext)VendorContextFactory.GetInstance().GetDBEFContext())
                {
                    var recordQuery = (from staff in vendorEFContext.StaffTable.AsNoTracking().Where(s => (s.CustomerID == companyID && s.State == CustomerAccountState.Active))
                                       join staffInfo in vendorEFContext.StaffInfoTable.AsNoTracking().Where(si => si.Status == DBItemState.Valid).DefaultIfEmpty()
                                       on staff.StaffID equals staffInfo.StaffID
                                       into ssi
                                       from item in ssi.DefaultIfEmpty()
                                       select new
                                       {
                                           Staff = staff,
                                           StaffInfo = item
                                       });
                    recordQuery = recordQuery.Where(r => r.Staff.Role == OperationLevel.Admin);
                    var recordList = recordQuery.ToList();
                    foreach (var record in recordList)
                    {
                        if (vendorMap.ContainsKey(record.Staff.StaffID))
                        {
                            if (record.StaffInfo != null)
                            {
                                vendorList[vendorMap[record.Staff.StaffID]].StaffInfoList.Add(record.StaffInfo);
                            }
                        }
                        else
                        {
                            vendorMap.Add(record.Staff.StaffID, vendorList.Count);
                            var vendor = new VendorModel()
                            {
                                StaffTable = record.Staff
                            };
                            if (record.StaffInfo != null)
                            {
                                vendor.StaffInfoList.Add(record.StaffInfo);
                            }
                            vendorList.Add(vendor);
                        }
                    }
                }
            }
            catch (DataException)
            {
                throw;
            }
            return PartialView("_ManageVendor", vendorList);
        }

        [HttpPost]
        [Authorize(Roles = "Staff, Admin, SysAdmin")]
        public JsonResult CheckPhoneNumberAvailable([Bind(Include = "Cellphone")]CCS_Staff_V1 StaffTable)
        {
            try
            {
                Guid customerId;
                if (!Guid.TryParse((string)Session["companyIDstr"], out customerId))
                {
                    customerId = new Guid(GetCurrentCustomerID());
                }
                var domainName = (GetCustomerInfo(customerId)).DomainName;
                var adAccount = string.Format("{0}@{1}", StaffTable.Cellphone, domainName);
                using (var vendorEFContext = (VendorContext)VendorContextFactory.GetInstance().GetDBEFContext())
                {
                    var record = (from staff in vendorEFContext.StaffTable.AsNoTracking().Where(s => (s.ADAccount == adAccount && s.State == CustomerAccountState.Active))
                                  select new
                                  {
                                      Staff = staff
                                  }).ToList();
                    return Json(record.Count == 0);
                }
            }
            catch (DataException)
            {
                return Json(false);
            }
        }


        [Authorize(Roles = "Staff, Admin, Vendor")]
        public ActionResult EditVendorProfile()
        {
            var vendor = GetCurrentStaff();
            if (vendor == null)
            {
                return View("Index");
                //throw new Exception(ErrorConstants.NoAssociatedCompany);
            }

            return View((object)GetProfilePictureUri(vendor.StaffID.ToString()));
        }

        [HttpPost]
        [Authorize(Roles = "Staff, Admin, Vendor")]
        public async Task<ActionResult> UploadVendorPicture(string staffidstr, HttpPostedFileBase uploadFile)
        {
            var nextView = "EditVendorProfile";
            if (ModelState.IsValid)
            {
                if (uploadFile == null || uploadFile.ContentLength == 0)
                {
                    ViewBag.UploadMsg = "Nothing to upload. Please choose a file.";
                }
                else
                {
                    // Blob path format: StaffID/yyyyMMddHHmmss
                    var dateTimeFormat = "yyyyMMddHHmmss";
                    var vendor = GetCurrentStaff();
                    if (vendor == null)
                    {
                        //throw new Exception(ErrorConstants.NoAssociatedCompany);
                    }
                    var staffID = string.IsNullOrWhiteSpace(staffidstr) ? vendor.StaffID.ToString() : staffidstr;
                    if (string.IsNullOrWhiteSpace(staffID))
                    {
                        return View("Index");
                    }
                    var blobName = String.Format("{0}/{1}", staffID, DateTime.UtcNow.ToString(dateTimeFormat));
                    var fileBlob = _vendorProfilePictureBlobContainer.GetBlockBlobReference(blobName);

                    // Upload local csv file to blob storage
                    using (var fileStream = uploadFile.InputStream)
                    {
                        await fileBlob.UploadFromStreamAsync(fileStream);
                    }
                    if (fileBlob == null)
                    {
                        ViewBag.UploadMsg = string.Format("Uploading file '{0}' is unsuccessful", uploadFile.FileName);
                    }
                    else
                    {
                        return this.Redirect(Request.UrlReferrer.ToString());
                    }
                }
            }
            return View(nextView);
        }


        /*Deprecated
        [HttpPost]
        [Authorize(Roles = "Staff, Admin, SysAdmin")]
        public async Task<ActionResult> OldRegister(
            [Bind(Include = "CustomerTable, StaffTable, Password, ComparePassword")]CustomerModel customer)
        {
            // for test
            try
            {
                if (ModelState.IsValid)
                {
                    // Fill default values
                    customer.CustomerTable.LastUpdatedDate = DateTime.Now;
                    // Set default customer region for now
                    customer.CustomerTable.Region = GeoRegion.Beijing;
                    customer.StaffTable.LastUpdatedDate = customer.CustomerTable.LastUpdatedDate;

                    // To do: Convert to phonetic symbol
                    customer.StaffTable.Name = customer.CustomerTable.Name;

                    var customerEFContext = (CustomerContext)CustomerContextFactory.GetInstance().GetDBEFContext();
                    customerEFContext.CustomerTable.Add(customer.CustomerTable);
                    customerEFContext.StaffTable.Add(customer.StaffTable);
                    if (string.IsNullOrWhiteSpace(customer.CustomerTable.Name) ||
                        string.IsNullOrWhiteSpace(customer.Password) ||
                        string.IsNullOrWhiteSpace(customer.ComparePassword))
                    {
                        // to do return webpage an error
                        return View(customer);
                    }
                    customer.StaffTable.geography = DbGeography.FromText(string.Format("POINT({1} {0})", customer.StaffTable.latitude, customer.StaffTable.longitude));
                    customerEFContext.SaveChanges();
                    // Customer ID is generated by DB
                    customer.StaffTable.ADAccount =
                        await CcsAdmin.Instance.CreateUserAsync(customer.StaffTable.StaffID.ToString(), customer.Password, null, OperationLevel.Vendor.ToString());

                    customer.StaffTable.ADDomain = CcsAdmin.Instance.DomainName;
                    customerEFContext.SaveChanges();
                    return View("RegisterResult", new MerchantRegisterUserModel { VendorAccountName = customer.StaffTable.ADAccount });
                }
            }
            catch (RetryLimitExceededException)
            {
                //Log the error (uncomment dex variable name and add a line here to write a log.)
                ModelState.AddModelError("", "Unable to save changes. Try again, and if the problem persists, see your system administrator.");
            }
            return View(customer);
        }*/

        //GET: VendorLocations
        public ActionResult ShowMap(double? CenterLatitude, double? CenterLongitude, string YunziIDs,
            int? StartIndex, int? Count, string ShowType, string Market, string MapStyle, string UserId, bool? NavigateOnLoad)
        {
            return RedirectToAction("ShowMap", "Map",
                new
                {
                    CenterLatitude = CenterLatitude,
                    CenterLongitude = CenterLongitude,
                    YunziIDs = YunziIDs,
                    StartIndex = StartIndex,
                    Count = Count,
                    ShowType = ShowType,
                    Market = Market,
                    MapStyle = MapStyle,
                    UserId = UserId,
                    NavigateOnLoad = NavigateOnLoad
                });
        }

        [Authorize(Roles = "Staff, Vendor")]
        public async Task<ActionResult> CreatePublicMsg(string blobName)
        {
            if (string.IsNullOrEmpty(blobName))
            {
                return View();
            }
            MerchantMsgModel adsMsgModel = null;
            try
            {
                var msgBlob = _adsBlobArchiveContainer.GetBlockBlobReference(blobName);
                string blobContent = await msgBlob.DownloadTextAsync();
                adsMsgModel = await Task.Factory.StartNew(() => JsonConvert.DeserializeObject<MerchantMsgModel>(blobContent));
            }
            catch (Exception)
            {
                //To do: 
            }
            return (adsMsgModel == null) ? View() : View(adsMsgModel);
        }


        [HttpPost]

        public JsonResult VendorSendPushNotif(string staffid)
        {
            string res = "";
            try
            {
                var usersInfo = _billDB.UserStaffFavoritesTable.Where(o => o.StaffID.ToString() == staffid && o.FavoriteStatus == FavoriteStatus.Favorite).ToList();
                if (usersInfo == null || usersInfo.Count() == 0)
                {


                    return new JsonResult() { Data = new { statuscode = 0, statusmsg = "Ooops, seems no user has added this vendor into favorites list." }, JsonRequestBehavior = JsonRequestBehavior.AllowGet };
                }
                var thumbprint = @"c76c6c7ef2cf139c922be1dbe5ab348b55b56533";
                var store = new X509Store(StoreName.My, StoreLocation.LocalMachine);
                store.Open(OpenFlags.OpenExistingOnly | OpenFlags.ReadOnly);
                int x = store.Certificates.Count;
                X509CertificateCollection col = store.Certificates;
                var certificate = store.Certificates
                    .Cast<X509Certificate2>()
                    .SingleOrDefault(c => string.Equals(c.Thumbprint, thumbprint, StringComparison.OrdinalIgnoreCase));
                var push = new PushBroker();
                //var appleCert = System.IO.File.ReadAllBytes(@"C:\Users\Hayk\Downloads\Outlook.com\CertUsersPasswordProtected.p12");

                var channel = new ApplePushChannelSettings(false, certificate.RawData, "pass@ios");
                push.RegisterAppleService(channel);
                Dictionary<string, string> deviceTokens = new Dictionary<string, string>();
                if (usersInfo != null)
                {
                    foreach (var userInfo in usersInfo)
                    {
                        var users = _billDB.UserTable.Where(o => o.UserID.ToString() == userInfo.UserID.ToString() && o.iOSDeviceToken != null);
                        if (users != null && users.Count() > 0)
                        {
                            foreach (var user in users)
                            {
                                //var user = users.First();
                                //CCS_User_V1 user = userInfo.UserTable;
                                string deviceToken = user.iOSDeviceToken;

                                if (!String.IsNullOrEmpty(deviceToken))
                                {
                                    res += deviceToken + " ";
                                    deviceTokens.Add(user.UserID.ToString(), deviceToken);

                                    push.QueueNotification(new AppleNotification()
                                                                .ForDeviceToken(deviceToken)
                                                                .WithAlert("A message from a vendor!!!")
                                                                .WithBadge(7));

                                }

                            }

                        }

                    }
                }
                else
                {
                    res += "usersinfo is NULL";
                }

                return new JsonResult() { Data = new { statuscode = 1, statusmsg = res }, JsonRequestBehavior = JsonRequestBehavior.AllowGet };
            }


            catch (Exception e)
            {
                if (e.InnerException != null)
                {
                    return new JsonResult() { Data = new { statuscode = 0, statusmsg = "Exception2: " + res + " " + e.InnerException.Message }, JsonRequestBehavior = JsonRequestBehavior.AllowGet };
                }
                else
                {
                    return new JsonResult() { Data = new { statuscode = 0, statusmsg = "Exception2: " + res + " " + e.Message }, JsonRequestBehavior = JsonRequestBehavior.AllowGet };

                }

            }




        }

        [HttpPost]
        [ValidateInput(false)]
        [Authorize(Roles = "Staff, Vendor")]
        public async Task<ActionResult> CreatePublicMsg(
            [Bind(Include = "Title, Author, Digest, Content, BlobName")]MerchantMsgModel adsMsgModel, string Button)
        {
            if (ModelState.IsValid && adsMsgModel != null)
            {
                try
                {
                    var blobcontent = await Task.Factory.StartNew(() => JsonConvert.SerializeObject(adsMsgModel));
                    // Blob path format: CustomerID/yyyy/MM/dd/HHmmss
                    var dateTimeFormat = "yyyy/MM/dd/HHmmss";
                    // to do: remove customerID hardcode 
                    var customerID = GetCurrentCustomerID();
                    var blobName = String.Format("{0}/{1}", customerID, DateTime.UtcNow.ToString(dateTimeFormat));
                    if (string.IsNullOrEmpty(adsMsgModel.BlobName))
                    {
                        adsMsgModel.BlobName = blobName;
                    }
                    var content = JsonConvert.SerializeObject(adsMsgModel);

                    if (Button == "Save")
                    {
                        var fileBlob = _adsBlobArchiveContainer.GetBlockBlobReference(adsMsgModel.BlobName);
                        await fileBlob.UploadTextAsync(content);
                    }
                    else if (Button == "Publish")
                    {
                        var fileBlob = _adsBlobContainer.GetBlockBlobReference(blobName);
                        await fileBlob.UploadTextAsync(content);
                    }
                }
                catch (Exception)
                {
                    //To do: 
                }
            }

            if (Button == "Save")
            {
                return RedirectToAction("ManageMsg");
            }
            return View();
        }

        [Authorize(Roles = "Staff, Vendor")]
        public async Task<ActionResult> ManageMsg()
        {
            // to do: remove customerID hardcode 
            string customerID = GetCurrentCustomerID();
            var msgList = new List<MerchantMsgModel>();
            string prefix = string.Format("{0}", customerID);
            foreach (IListBlobItem item in _adsBlobArchiveContainer.ListBlobs(prefix, true))
            {
                var blob = (CloudBlockBlob)item;
                string blobContent = await blob.DownloadTextAsync();
                MerchantMsgModel msgModel = JsonConvert.DeserializeObject<MerchantMsgModel>(blobContent);
                msgList.Add(msgModel);
            }

            return View(msgList);
        }

        [Authorize(Roles = "Staff")]
        public ActionResult UploadBill(string uploadMsg)
        {
            ViewBag.UploadMsg = uploadMsg;
            var customerId = GetCurrentCustomerID();
            var query = new TableQuery<BillTemplateTableEntity>().Where(
                TableQuery.GenerateFilterCondition("PartitionKey",
                QueryComparisons.Equal, customerId));
            var billTypeList = new List<string>();
            var billTemplateEntityList = new List<BillTemplateTableEntity>();
            foreach (var entity in _billTemplateTable.ExecuteQuery(query))
            {
                billTypeList.Add(entity.RowKey);
                billTemplateEntityList.Add(entity);
            }
            billTypeList.Add(CCSPrototypeCommon.Constants.NewBillTypeDeclaration);
            Session["BillTemplateTableEntity"] = billTemplateEntityList;
            var list = new SelectList(billTypeList);
            ViewBag.billTypeList = list;
            return View();
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        [Authorize(Roles = "Staff")]
        public async Task<ActionResult> UploadBill(string billType, HttpPostedFileBase uploadFile)
        {
            string nextView = "UploadBill";

            if (ModelState.IsValid)
            {
                if (uploadFile == null || uploadFile.ContentLength == 0)
                {
                    ViewBag.UploadMsg = "Nothing to upload. Please choose a file.";
                }
                else
                {
                    if (billType.Equals(CCSPrototypeCommon.Constants.NewBillTypeDeclaration))
                    {
                        billType = string.Empty;
                    }
                    // Blob path format: CustomerID/yyyy/MMdd/HHmmss.fileExtension
                    string dateTimeFormat = "yyyy/MMdd/HHmmss";
                    // to do: remove customerID hardcode 
                    var customerID = GetCurrentCustomerID();
                    string blobName = String.Format("{0}/{1}{2}", customerID, DateTime.UtcNow.ToString(dateTimeFormat), Path.GetExtension(uploadFile.FileName));
                    CloudBlockBlob fileBlob = _billBlobContainer.GetBlockBlobReference(blobName);

                    // Upload local csv file to blob storage
                    using (var fileStream = uploadFile.InputStream)
                    {
                        await fileBlob.UploadFromStreamAsync(fileStream);
                    }
                    if (fileBlob == null)
                    {
                        ViewBag.UploadMsg = string.Format("Uploading file '{0}' is unsuccessful", uploadFile.FileName);
                    }
                    else
                    {
                        if (Session["BillTemplateTableEntity"] == null)
                        {
                            return RedirectToAction(nextView, new { uploadMsg = "Please try again" });
                        }
                        var billTemplateEntityList = (List<BillTemplateTableEntity>)Session["BillTemplateTableEntity"];
                        string[] columns = null;
                        using (Stream fs = fileBlob.OpenRead())
                        {
                            StreamReader csv = new StreamReader(fs);
                            // Header
                            string line = csv.ReadLine();
                            columns = line.Split(',');
                        }
                        if (columns == null)
                        {
                            ViewBag.UploadMsg = "This file is not supported";
                        }
                        try
                        {
                            var billTemplate = Utility.GetTemplateEntity(billType, billTemplateEntityList);
                            var billTemplateMapping = Utility.GetColumnMappingByHeader(new List<string>(columns), billTemplate.GetColumnList());
                        }
                        catch (BillDataException billExp)
                        {
                            var billTemplateModel = new BillTemplateModel()
                            {
                                BillType = billType,
                                BillHeaderList = Utility.GetCsvTableHeader(columns),
                                BillTemplateColumnList = BillTemplateTableEntity.GetColumnDefinationList()
                            };
                            ViewBag.UploadMsg = billExp;
                            Session["uploadBlobName"] = blobName;
                            nextView = "SetBillTemplate";
                            return View(nextView, billTemplateModel);
                        }
                        var queueMsg = new BillQueueMsg()
                        {
                            BillType = billType,
                            BlobName = blobName
                        };
                        nextView = "UploadBill";
                        var queueMsgString = await Task.Factory.StartNew(() => JsonConvert.SerializeObject(queueMsg));
                        CloudQueueMessage queueMessage = new CloudQueueMessage(queueMsgString);
                        await _billQueue.AddMessageAsync(queueMessage);
                        return RedirectToAction(nextView, new { uploadMsg = "File uploaded" });
                    }
                }
            }
            return View(nextView);
        }


        [Authorize(Roles = "Staff")]
        public ActionResult UploadMeterData(string uploadMsg)
        {
            ViewBag.uploadMsg = uploadMsg;
            return View();
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        [Authorize(Roles = "Staff")]
        public async Task<ActionResult> UploadMeterData(HttpPostedFileBase uploadFile)
        {
            string nextView = "UploadMeterData";

            if (ModelState.IsValid)
            {
                if (uploadFile == null || uploadFile.ContentLength == 0)
                {
                    ViewBag.UploadMsg = "Nothing to upload. Please choose a file.";
                }
                else
                {
                    // Blob path format: CustomerID/yyyy/MMdd/HHmmss.fileExtension
                    string dateTimeFormat = "yyyy/MMdd/HHmmss";
                    // to do: remove customerID hardcode 
                    var customerID = GetCurrentCustomerID();
                    const string prefix = "Raw Data";
                    string blobName = String.Format("{0}-{1}/{2}{3}", prefix, customerID, DateTime.UtcNow.ToString(dateTimeFormat), Path.GetExtension(uploadFile.FileName));
                    CloudBlockBlob fileBlob = _rawDataBlobContainer.GetBlockBlobReference(blobName);

                    // Upload local csv file to blob storage
                    using (var fileStream = uploadFile.InputStream)
                    {
                        await fileBlob.UploadFromStreamAsync(fileStream);
                    }

                    if (fileBlob == null)
                    {
                        ViewBag.UploadMsg = string.Format("Uploading file '{0}' is unsuccessful", uploadFile.FileName);
                    }
                    else
                    {
                        string[] columns = null;
                        using (Stream fs = fileBlob.OpenRead())
                        {
                            StreamReader csv = new StreamReader(fs);
                            // Get header
                            string line = csv.ReadLine();
                            columns = line.Split(',');
                        }
                        if (columns == null)
                        {
                            ViewBag.UploadMsg = "This file is not supported";
                        }
                        else
                        {
                            RawDataQueueMsg queueMsg = new RawDataQueueMsg()
                            {
                                MeasureType = null,  // Reserved for now
                                BlobName = blobName
                            };
                            var queueMsgString = await Task.Factory.StartNew(() => JsonConvert.SerializeObject(queueMsg));
                            CloudQueueMessage queueMessage = new CloudQueueMessage(queueMsgString);
                            await _rawDataQueue.AddMessageAsync(queueMessage);
                            return RedirectToAction(nextView, new { uploadMsg = "File uploaded" });
                        }
                        
                    }
                }
            }
            return View(nextView);
        }


        [HttpPost]
        public JsonResult UploadRawDataMobile()
        {
            string data = "NA";
            string responseMsg = "NA";
            try
            {
                //data = new System.IO.StreamReader(Request.InputStream).ReadToEnd();
                System.IO.Stream stream = Request.InputStream;
                // Blob path format: CustomerID/yyyy/MMdd/HHmmss.fileExtension
                string dateTimeFormat = "yyyy/MMdd/HHmmss";
                // to do: remove customerID hardcode 
                var customerID = "EBCTest";
                const string prefix = "Raw Data";
                string blobName = String.Format("{0}-{1}/{2}{3}", prefix, customerID, DateTime.Now.ToString(dateTimeFormat), "Filename");
                CloudBlockBlob fileBlob = _rawDataBlobContainer.GetBlockBlobReference(blobName);
                responseMsg += " Uploading " + blobName;
                // Upload local csv file to blob storage
               
                using (var fileStream = stream)
                {
                    responseMsg += " start ";
                    fileBlob.UploadFromStream(fileStream);
                }
                responseMsg += " done ";
                if (fileBlob == null)
                {
                    responseMsg += " fileblob is null ";
                }
                else
                {
                    responseMsg += " fileblob is NOT null ";

                    string[] columns = null;
                    using (Stream fs = fileBlob.OpenRead())
                    {
                        StreamReader csv = new StreamReader(fs);
                        // Get header
                        string line = csv.ReadLine();
                        columns = line.Split(',');
                    }
                    if (columns == null)
                    {
                        responseMsg += " This file is not supported ";
                    }
                    else
                    {
                        responseMsg += " This file is  supported ";

                        RawDataQueueMsg queueMsg = new RawDataQueueMsg()
                        {
                            MeasureType = null,  // Reserved for now
                            BlobName = blobName
                        };
                        var queueMsgString = JsonConvert.SerializeObject(queueMsg);
                        CloudQueueMessage queueMessage = new CloudQueueMessage(queueMsgString);
                        _rawDataQueue.AddMessageAsync(queueMessage);
                        responseMsg += " File uploaded ";
                    }

                    
                }

            }
            catch(Exception ex)
            {
                return new JsonResult() { Data = new { statusCode = "Sucess", statusMsg = ex.Message + " " + responseMsg +" " + data }, JsonRequestBehavior = JsonRequestBehavior.AllowGet };
            }
            return new JsonResult() { Data = new { statusCode = "Sucess", statusMsg = responseMsg + " " + data }, JsonRequestBehavior = JsonRequestBehavior.AllowGet };
        }



        [Authorize(Roles = "Staff")]
        public async Task<ActionResult> DownloadBill()
        {
            return View();
        }

        [Authorize(Roles = "Staff")]
        public async Task<ActionResult> Download(string month, string BlobUri)
        {
            try
            {
                CloudBlockBlob blob = new CloudBlockBlob(new Uri(BlobUri));

                using (MemoryStream memStream = new MemoryStream())
                {
                    await blob.DownloadToStreamAsync(memStream);

                    return new FileContentResult(memStream.ToArray(), blob.Properties.ContentType)
                    {
                        FileDownloadName = month + BlobUri.Substring(BlobUri.LastIndexOf('-') + 1)
                    };
                }
            }
            catch (Exception)
            {
                return new HttpNotFoundResult();
            }

        }

        [Authorize(Roles = "Staff")]
        public ActionResult DownloadFileFromBlob()
        {
            string view = "HomeWaterCompanyPaymentDownload";

            DateTime time = DateTime.UtcNow;
            foreach (IListBlobItem blob in paymentsBlobContainer.ListBlobs(null, false))
            {
                Uri blobUri;
                string curMonth = time.ToString("yyyy-MM");
                if (!BlobHashMap.TryGetValue(curMonth, out blobUri))
                {
                    BlobHashMap.Add(curMonth, blob.Uri);
                }

                time = time.AddMonths(-1);
            }

            return View(view, BlobHashMap);
        }

        /*
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<ActionResult> UploadFileToBlob(HttpPostedFileBase uploadFile)
        {
            CloudBlockBlob fileBlob = null;
            string view = "HomeWaterCompany";

            if (ModelState.IsValid)
            {
                if (uploadFile == null || uploadFile.ContentLength == 0)
                {
                    ViewBag.UploadMsg = "Nothing to upload. Please choose a file.";
                }
                else
                {
                    // Blob path format: CustomerID/yyyy/MMdd/HHmmss.fileExtension
                    string dateTimeFormat = "yyyy/MMdd/HHmmss";
                    // to do: remove customerID hardcode 
                    string blobName = String.Format("{0}/{1}{2}", "E1581298-5894-44EE-8FE3-7017D09B206B", DateTime.UtcNow.ToString(dateTimeFormat), Path.GetExtension(uploadFile.FileName));
                    fileBlob = filesBlobContainer.GetBlockBlobReference(blobName);

                    // Upload local csv file to blob storage
                    using (var fileStream = uploadFile.InputStream)
                    {
                        await fileBlob.UploadFromStreamAsync(fileStream);
                    }

                    if (fileBlob == null)
                    {
                        ViewBag.UploadMsg = string.Format("Uploading file '{0}' is unsuccessful", uploadFile.FileName);
                    }
                    else
                    {
                        string fileURL = fileBlob.Uri.ToString();

                        // Add a message to message queue
                        CloudQueueMessage queueMessage = new CloudQueueMessage(blobName);
                        await filesQueue.AddMessageAsync(queueMessage);

                        ViewBag.UploadMsg = string.Format("File '{0}' is uploaded to blob '{1}' @ '{2}'", uploadFile.FileName, blobName, fileURL);
                        view = "HomeWaterCompanyBillDisplay";
                    }

                }
            }
            return View(view);
        }*/





        [HttpPost]
        public ActionResult AddCustomizeInfo(int? index)
        {
            ViewBag.CustomizeInfoID = Guid.NewGuid().ToString();
            return PartialView("_CustomizeInfo");
        }

        [HttpPost]
        public PartialViewResult RemoveCustomizeInfo(int? index)
        {
            return PartialView("_CustomizeInfo");
        }
        [ValidateAntiForgeryToken]
        public async Task<ActionResult> SetBillTemplate(FormCollection collection, string Button)
        {
            var nextView = "UploadBill";
            if (Session["uploadBlobName"] == null || Session["BillTemplateTableEntity"] == null)
            {
                return RedirectToAction(nextView, new { uploadMsg = "Please try again" });
            }
            if (Button.Equals("Submit"))
            {
                //to do: remove customer id hardcode
                var customerID = GetCurrentCustomerID();
                var billType = collection["BillType"];
                if (string.IsNullOrWhiteSpace(billType) || billType.Equals(CCSPrototypeCommon.Constants.NewBillTypeDeclaration))
                {
                    ViewBag.UploadMsg = "Invalid bill type";
                    return RedirectToAction(nextView, new { uploadMsg = "To do: redirect to set bill template page" });
                }

                var billTemplateEntity = new BillTemplateTableEntity(customerID, billType)
                {
                    AmountDueColumn = collection["AmountDue"],
                    UserNameColumn = collection["UserName"],
                    UserAddressColumn = collection["UserAddress"],
                    CellphoneColumn = collection["Cellphone"]
                };
                var insertOperation = TableOperation.InsertOrReplace(billTemplateEntity);
                await _billTemplateTable.ExecuteAsync(insertOperation);

                var blobName = (string)Session["uploadBlobName"];
                var queueMsg = new BillQueueMsg()
                {
                    BillType = billType,
                    BlobName = blobName
                };
                var queueMsgString = await Task.Factory.StartNew(() => JsonConvert.SerializeObject(queueMsg));
                CloudQueueMessage queueMessage = new CloudQueueMessage(queueMsgString);
                await _billQueue.AddMessageAsync(queueMessage);
                return RedirectToAction(nextView, new { uploadMsg = "File uploaded" });
            }
            return RedirectToAction(nextView);
        }

        // POST: Merchant/Create
        //[HttpPost]
        public ActionResult CreateTestData(string name,
            string YunziID, string region, double? latitude, double? longitude, string address, string password, int? Count)
        {
            try
            {
                Random random = new Random(DateTime.UtcNow.DayOfYear);
                double templatitude, templongitude;
                double flag = -1;
                string YunziPrefix = "SN" + (random.NextDouble() * 100).ToString("F0");
                int DataCount = Count ?? 1;
                if (DataCount > 500)
                {
                    DataCount = 500;
                }
                //for (int i = 0; i < DataCount; i++)
                //{
                //flag = -1 * flag;
                //latitude = random.NextDouble() * 0.1 * flag * i + (CenterLatitude ?? 47.742162);
                //longitude = random.NextDouble() * 0.1 * flag * i + (CenterLongitude ?? -122.192583);

                if (latitude == null)
                {
                    templatitude = random.NextDouble() * 0.1 * flag + 47.742162;
                }
                else
                {
                    templatitude = latitude ?? 47.742162;
                }

                if (longitude == null)
                {
                    templongitude = random.NextDouble() * 0.1 * flag - 122.192583;
                }
                else
                {
                    templongitude = longitude ?? -122.192583;
                }

                CustomerModel customer = new CustomerModel();
                customer.CustomerTable = new CCS_Customer_V1();
                customer.CustomerTable.Name = name;
                customer.CustomerTable.Region = GetRegion(region);
                customer.CustomerTable.State = CustomerAccountState.Active;

                customer.StaffTable = new CCS_Staff_V1();
                customer.StaffTable.geography = DbGeography.FromText(string.Format("POINT({1} {0})", templatitude, templongitude));
                customer.StaffTable.Address = address;
                customer.StaffTable.latitude = templatitude;
                customer.StaffTable.longitude = templongitude;

                customer.Password = string.IsNullOrEmpty(password) ? "1qaz@WSX" : password;
                customer.ComparePassword = customer.Password;

                customer.StaffTable.SNyunzi = string.IsNullOrEmpty(YunziID) ? (YunziPrefix + (random.NextDouble() * 100).ToString("X4")) : YunziID;
                //OldRegister(customer);
                //}
            }
            catch
            {
                ;
            }
            return RedirectToAction("ShowMap");
        }

        public ActionResult QueryBill()
        {
            return this.RedirectToAction("QueryFiltersUI", "Merchant", new { queryType = "BillQuery"});
        }

        // All the Bill Query methods now live in class QueryableController

        // Show default page
        public ActionResult ToDo(string PageTitle, string Details)
        {
            ViewBag.Title = PageTitle;
            ViewBag.ToDoDetails = Details;
            return View();
        }

    }

    // Merchant - Vendor
    public partial class MerchantController
    {
        [Authorize(Roles = "Staff, Vendor, Gatherer")]
        public ActionResult CheckUserBill()
        {

            return View();
        }

        [Authorize(Roles = "Staff, Vendor, Gatherer")]
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult ViewUserBill(
            [Bind(Include = "UserInputAccountID")] UserModelHome userModelHome)
        {
            var userInputID = userModelHome.UserInputAccountID;
            var billContex = (BillContext)BillContextFactory.GetInstance().GetDBEFContext();
            var billsWithThisUser = BillCommon.getCCSBillListWithInputID(billContex, userInputID);
            if (billsWithThisUser == null || billsWithThisUser.Count() == 0)
            {
                return RedirectToAction("GeneralError", "Error", new
                {
                    message = string.Format("Did not find any bills from the user:{0}", userInputID),
                    linkText = "try another user",
                    act = "CheckUserBill",
                    ctrl = "Merchant"
                });
            }
            return View("ShowUserBill", billsWithThisUser);
        }
    }
}
