﻿using CCSAdmin;
using CCSPrototypeCommon;
using CCSPrototypeCommon.Query;
using HomeWeb.Models;
using Microsoft.WindowsAzure.ServiceRuntime;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.Storage.Blob;
using Microsoft.WindowsAzure.Storage.Queue;
using Microsoft.WindowsAzure.Storage.RetryPolicies;
using Microsoft.WindowsAzure.Storage.Table;
using Newtonsoft.Json;
using PagedList;
using System;
using System.Collections.Generic;
using System.Data;
using System.Data.Entity.Infrastructure;
using System.Data.Entity.Spatial;
using System.Data.Entity.Validation;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Threading.Tasks;
using System.Web;
using System.Web.Mvc;

namespace HomeWeb.Controllers
{
    public class QueryableController : BaseController
    {
        protected CloudBlobContainer queryBlobContainer;
        protected CloudQueue queryDownloadQueue;
        protected CloudQueue queryInitiateQueue;

        protected Dictionary<string, BaseFilter> filters = null;

        protected QueryType query = QueryType.Undefined;

        // Max 6 records on each page
        protected static int PageList_PageSize = 6;

        public QueryableController()
        {
            InitializeCloudStorage();
        }

        protected virtual void InitializeCloudStorage()
        {
            CloudStorageAccount storageAccount = CloudStorageAccount.Parse(RoleEnvironment.GetConfigurationSettingValue("StorageConnectionString"));
            CloudBlobClient blobClient = storageAccount.CreateCloudBlobClient();
            blobClient.DefaultRequestOptions.RetryPolicy = new LinearRetry(TimeSpan.FromSeconds(3), 3);

            queryBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.QueryStorageName);
            queryBlobContainer.CreateIfNotExists();

            CloudQueueClient queueClient = storageAccount.CreateCloudQueueClient();

            queryInitiateQueue = queueClient.GetQueueReference(CCSPrototypeCommon.Constants.QueryInitStorageName);
            queryInitiateQueue.CreateIfNotExists();
            queryDownloadQueue = queueClient.GetQueueReference(CCSPrototypeCommon.Constants.QueryDownloadStorageName);
            queryDownloadQueue.CreateIfNotExists();

        }


        public ActionResult Query()
        {
            ViewBag.CustomizeInfoNum = (Session["QueryConditionNum"] != null) ? Session["QueryConditionNum"] : Session["QueryConditionNum"] = 1;
            return View();
        }


        [HttpPost]
        public ActionResult Query(
            [Bind(Include = "ConditionList")]QueryModel queryData)
        {
            return View(queryData);
        }


        // The first query action method for a controller

        //[OutputCache(NoStore = true, Duration = 0, VaryByParam = "None")]
        public ActionResult QueryFiltersUI(string queryType)
        {

            filters = (Session["filters"] != null) ? ((Dictionary<string, BaseFilter>)Session["filters"]) : (Dictionary<string, BaseFilter>)(Session["filters"] = new Dictionary<string, BaseFilter>());
            BaseFilter bs = new BaseFilter();
            bs.searchModel = new BlankFilter(bs.BaseSearchID);
            filters[bs.BaseSearchID.ToString()] = bs;
            Session["filters"] = filters;

            query = Utility.ParseEnum<QueryType>(queryType);

            bs.ListBaseTypes.AddRange(FillQueryBaseTypeList(query));

            return View(bs);

        }

        // The second query action method for a controller

        //[OutputCache(NoStore = true, Duration = 0, VaryByParam = "None")]
        public ActionResult QueryFiltersUI1(string queryType)
        {

            filters = (Session["filters"] != null) ? ((Dictionary<string, BaseFilter>)Session["filters"]) : (Dictionary<string, BaseFilter>)(Session["filters"] = new Dictionary<string, BaseFilter>());
            BaseFilter bs = new BaseFilter();
            bs.searchModel = new BlankFilter(bs.BaseSearchID);
            filters[bs.BaseSearchID.ToString()] = bs;
            Session["filters"] = filters;

            query = Utility.ParseEnum<QueryType>(queryType);

            bs.ListBaseTypes.AddRange(FillQueryBaseTypeList(query));

            return View(bs);

        }

        //[HttpPost]
        //public ActionResult BillChange(string billType)
        //{

        //    filters = (Session["filters"] != null) ? ((Dictionary<string, BaseFilter>)Session["filters"]) : (Dictionary<string, BaseFilter>)(Session["filters"] = new Dictionary<string, BaseFilter>());

        //    IEnumerable<PropertyInfo> properties = GetProperties();
        //    List<string> columns = new List<string>();
        //    foreach (var prop in properties)
        //    {
        //        columns.Add(prop.Name);
        //    }

        //    BaseFilter bf = new BaseFilter();
        //    bf.searchModel = new BlankFilter(bf.BaseSearchID);
        //    bf.SelectedBaseType = billType;
        //    bf.ListColumns.AddRange(columns);
        //    filters[bf.BaseSearchID.ToString()] = bf;
        //    Session["filters"] = filters;

        //    return PartialView("~/Views/Merchant/_FilterCondition.cshtml", bf);

        //}

        public ActionResult BaseTypeChange(string billType)
        {

            filters = (Session["filters"] != null) ? ((Dictionary<string, BaseFilter>)Session["filters"]) : (Dictionary<string, BaseFilter>)(Session["filters"] = new Dictionary<string, BaseFilter>());

            IEnumerable<PropertyInfo> properties = GetProperties(QueryType.BillQuery);
            List<string> columns = new List<string>();
            foreach (var prop in properties)
            {
                columns.Add(prop.Name);
            }

            BaseFilter bf = new BaseFilter();
            bf.searchModel = new BlankFilter(bf.BaseSearchID);
            bf.SelectedBaseType = billType;
            bf.ListColumns.AddRange(columns);
            filters[bf.BaseSearchID.ToString()] = bf;
            Session["filters"] = filters;

            return PartialView("_FilterCondition", bf);
        }


        //Filters
        [HttpPost]
        public ActionResult FilterChange(string queryType, string filterName, string filterValue)
        {
            filters = (Session["filters"] != null) ? ((Dictionary<string, BaseFilter>)Session["filters"]) : (Dictionary<string, BaseFilter>)(Session["filters"] = new Dictionary<string, BaseFilter>());
            BaseFilter bs = null;

            query = Utility.ParseEnum<QueryType>(queryType);

            if (filters.TryGetValue(filterName, out bs))
            {
                if (String.IsNullOrEmpty(filterValue))
                {
                    bs.searchModel = new BlankFilter(bs.BaseSearchID);
                    return PartialView("~/Views/Shared/EditorTemplates/BlankFilter.cshtml", bs.searchModel);
                }
                else
                {
                    //string billType = bs.SelectedBaseType;
                    string column = filterValue;
                    //string dataType = _billEntity.BillTemplateTable.Where(obj => obj.BillTypeID == billType && obj.Column == column).Select(dt => dt.ColumnType).FirstOrDefault();

                    string dataType = TryGetDataTypeName(query, column, "string");

                    if (String.Equals(dataType, "integer", StringComparison.OrdinalIgnoreCase) ||
                        String.Equals(dataType, "double", StringComparison.OrdinalIgnoreCase))
                    {
                        bs.searchModel = new NumericFilter(bs.BaseSearchID);
                        bs.searchModel.Property = column;
                        bs.searchModel.TargetTypeName = GetAssemblyQualifiedName();
                        return PartialView("~/Views/Shared/EditorTemplates/NumericFilter.cshtml", bs.searchModel);
                    }
                    else if (String.Equals(dataType, "datetime", StringComparison.OrdinalIgnoreCase))
                    {
                        bs.searchModel = new DateFilter(bs.BaseSearchID);
                        bs.searchModel.Property = column;
                        bs.searchModel.TargetTypeName = GetAssemblyQualifiedName();
                        return PartialView("~/Views/Shared/EditorTemplates/DateFilter.cshtml", bs.searchModel);
                    }
                    else
                    {
                        bs.searchModel = new TextFilter(bs.BaseSearchID);
                        bs.searchModel.Property = column;
                        bs.searchModel.TargetTypeName = GetAssemblyQualifiedName();
                        return PartialView("~/Views/Shared/EditorTemplates/TextFilter.cshtml", bs.searchModel);
                    }
                }
            }

            return null;

        }

        //public T ParseEnum<T>(string value)
        //{
        //    return (T)Enum.Parse(typeof(T), value, true);
        //}

        protected string TryGetDataTypeName(QueryType queryType, string PropertyName, string DefaultType)
        {
            string DataType = DefaultType;
            if (string.IsNullOrEmpty(PropertyName))
            {
                return DataType;
            }
            IEnumerable<PropertyInfo> PList = GetProperties(queryType);

            foreach (PropertyInfo p in PList)
            {
                if (PropertyName.Equals(p.Name, StringComparison.OrdinalIgnoreCase))
                {
                    DataType = p.PropertyType.Name;
                    break;
                }
            }

            return DataType;
        }


        [HttpPost]
        public PartialViewResult AddFilterCondition(string queryType)
        {
            filters = (Session["filters"] != null) ? ((Dictionary<string, BaseFilter>)Session["filters"]) : (Dictionary<string, BaseFilter>)(Session["filters"] = new Dictionary<string, BaseFilter>());

            query = Utility.ParseEnum<QueryType>(queryType);

            IEnumerable<PropertyInfo> properties = GetProperties(query);
            List<string> columns = new List<string>();
            columns.Add("请选择查询字段");
            foreach (var prop in properties)
            {
                columns.Add(prop.Name);
            }

            BaseFilter bf = new BaseFilter();
            bf.searchModel = new BlankFilter(bf.BaseSearchID);
            //bf.SelectedBillType = billType;
            bf.ListColumns.AddRange(columns);
            filters[bf.BaseSearchID.ToString()] = bf;
            Session["filters"] = filters;

            return PartialView("_QueryFiltersUICondition", bf);
        }



        [HttpPost]
        public PartialViewResult RemoveQueryCondition(int? index)
        {
            return null;
        }


        protected IEnumerable<T> FilterApplied<T>(FormCollection collection) where T : class
        {

            Dictionary<string, List<string>> dt = new Dictionary<string, List<string>>();


            foreach (string key in collection.AllKeys)
            {
                string guid = key.Split('_')[0];
                if (key.EndsWith("SelectedCategory") && !dt.ContainsKey(guid))
                {
                    dt.Add(guid, new List<string>());
                }

            }

            foreach (string guid in dt.Keys)
            {

                foreach (string key in collection.AllKeys)
                {
                    if (key.StartsWith(guid))
                    {
                        int index = key.IndexOf("_");
                        string str1 = key.Substring(0, index);
                        string str2 = key.Substring(index + 1, key.Length - index - 1);
                        dt[str1].Add(str2);
                    }
                }
            }

            List<AbstractFilter> searchCriteria = new List<AbstractFilter>();
            //var guids = collection.AllKeys.Select(s => collection[s]).Where(s => s.EndsWith("DropDownList")).ToList();
            foreach (string guid in dt.Keys)
            {
                AbstractFilter searchCriterion = null;
                string modelType = collection[guid + "_ModelTypeName"];

                if (string.IsNullOrEmpty(modelType))
                {
                    continue;
                }

                if (Type.GetType(modelType) == typeof(NumericFilter))
                {
                    searchCriterion = new NumericFilter();
                }
                else if (Type.GetType(modelType) == typeof(DateFilter))
                {
                    searchCriterion = new DateFilter();
                }
                else if (Type.GetType(modelType) == typeof(TextFilter))
                {
                    searchCriterion = new TextFilter();
                }

                foreach (string property in dt[guid])
                {
                    try
                    {
                        PropertyInfo pi = searchCriterion.GetType().GetProperty(property);
                        if (pi.PropertyType == typeof(NumericComparators) || pi.PropertyType == typeof(int) || pi.PropertyType == typeof(int?))
                        {
                            pi.SetValue(searchCriterion, Int32.Parse(collection[guid + "_" + property]));
                        }
                        else if (pi.PropertyType == typeof(Double) || pi.PropertyType == typeof(Double?))
                        {
                            pi.SetValue(searchCriterion, Double.Parse(collection[guid + "_" + property]));
                        }
                        else if (pi.PropertyType == typeof(string))
                        {
                            pi.SetValue(searchCriterion, collection[guid + "_" + property]);
                        }
                        else if (pi.PropertyType == typeof(DateTime) || pi.PropertyType == typeof(DateTime?))
                        {
                            try
                            {
                                pi.SetValue(searchCriterion, DateTime.Parse(collection[guid + "_" + property]));
                            }
                            catch (Exception e)
                            {
                                pi.SetValue(searchCriterion, new DateTime(2015, 1, 1));
                            }
                        }
                    }
                    catch (FormatException e)
                    {
                        return null;
                    }
                }

                if (searchCriterion != null)
                {
                    searchCriteria.Add(searchCriterion);
                }

                // Save filters
                Session["filterCriteria"] = searchCriteria;
            }

            IEnumerable<T> ResultList = GetQueryResult<T>(collection, searchCriteria, query);

            Session["billType"] = collection["billType"] as string;
            Session["filterResults"] = ResultList;

            return ResultList;
        }



        #region All these Virtual methods must be overridden in derived class. Otherwise, the Bill Query will be used.
        protected virtual string GetAssemblyQualifiedName()
        {
            switch (query)
            {
                case QueryType.MeterProfileQuery:
                    return typeof(CCS_MeterData_V1).AssemblyQualifiedName;
                case QueryType.RawDataQuery:
                    return typeof(CCS_RawData_V1).AssemblyQualifiedName;
                case QueryType.BillQuery:
                    return typeof(CCS_Bill_Joined_V1).AssemblyQualifiedName;
                case QueryType.PaymentQuery:
                    return typeof(CCS_Payment_V1).AssemblyQualifiedName;
                default:
                    throw new Exception("Unsupported query type: " + query.ToString());
            }
        }


        protected virtual IEnumerable<string> FillQueryBaseTypeList(QueryType queryType)
        {
            List<string> result = new List<string>();
            List<PropertyInfo> properties = new List<PropertyInfo>();
            RawDataContext rawDataDB = (RawDataContext)RawDataContextFactory.GetInstance().GetDBEFContext();
            switch (queryType)
            {
                case QueryType.MeterProfileQuery:
                    properties.AddRange(typeof(CCS_MeterData_V1).GetProperties());
                    foreach (var prop in properties)
                    {
                        result.Add(prop.Name);
                    }
                    break;

                case QueryType.RawDataQuery:
                    properties.AddRange(typeof(CCS_RawData_V1).GetProperties());
                    foreach (var prop in properties)
                    {
                        result.Add(prop.Name);
                    }
                    break;

                case QueryType.BillQuery:
                    BillContext billDB = (BillContext)BillContextFactory.GetInstance().GetDBEFContext();
                    result = billDB.BillTemplateTable.Select(r => r.BillType).Distinct().ToList();
                    break;

                case QueryType.PaymentQuery:
                    result.Add("用户缴费历史");
                    result.Add("代缴点缴费历史");
                    result.Add("催缴员缴费历史");
                    break;

                default:
                    throw new Exception("Unsupported query type: " + queryType);
            }

            return result;

        }


        protected virtual IEnumerable<PropertyInfo> GetProperties(QueryType queryType)
        {
            List<PropertyInfo> result = new List<PropertyInfo>();

            switch (queryType)
            {
                case QueryType.MeterProfileQuery:
                    result.AddRange(typeof(CCS_MeterData_V1).GetProperties());
                    break;
                case QueryType.RawDataQuery:
                    result.AddRange(typeof(CCS_RawData_V1).GetProperties());
                    break;
                case QueryType.BillQuery:
                    result.AddRange(typeof(CCS_Bill_V1).GetProperties());
                    result.AddRange(typeof(CCS_User_V1).GetProperties());
                    break;
                case QueryType.PaymentQuery:
                    break;
                default:
                    throw new Exception("Unsupported query type " + queryType.ToString());
            }
            return result;
        }

        //
        // Query Bill methods
        //
        [HttpPost]
        public virtual ActionResult QueryBillResult(FormCollection collection)
        {
            query = QueryType.BillQuery;

            IEnumerable<CCS_Bill_Joined_V1> result = FilterApplied<CCS_Bill_Joined_V1>(collection);

            int pageNumber = 1;

            //return View("FilterResults", paginatedResult.Result.ToPagedList(pageNumber, pageSize));

            //var x = PartialView("FilterResults", result.ToList());
            //var x = PartialView("FilterResults", result.ToPagedList(pageNumber, pageSize));
            return PartialView("_QueryResults", result.ToPagedList(pageNumber, PageList_PageSize));
        }

        public virtual ActionResult QueryBillResultPaged(int? page)
        {
            query = QueryType.BillQuery;

            // default type is CCS_Bill_Joined_V1
            IEnumerable<CCS_Bill_Joined_V1> result = (IEnumerable<CCS_Bill_Joined_V1>)Session["filterResults"];

            //PaginatedList<Bill> paginatedResult = new PaginatedList<Bill>(result, page ?? 0, null);

            int pageNumber = page ?? 1;

            //return View("FilterResults", paginatedResult.Result.ToPagedList(pageNumber, pageSize));

            return PartialView("_QueryResults", result.ToPagedList(pageNumber, PageList_PageSize));
        }

        //
        // Query meter profile methods
        //
        [HttpPost]
        public virtual ActionResult QueryProfile(FormCollection collection)
        {
            query = QueryType.MeterProfileQuery;

            IEnumerable<CCS_MeterData_V1> result = FilterApplied<CCS_MeterData_V1>(collection);

            int pageNumber = 1;

            return PartialView("_QueryResults", result.ToPagedList(pageNumber, PageList_PageSize));
        }

        public virtual ActionResult QueryProfilePaged(int? page)
        {
            query = QueryType.MeterProfileQuery;

            IEnumerable<CCS_MeterData_V1> result = (IEnumerable<CCS_MeterData_V1>)Session["filterResults"];

            int pageNumber = page ?? 1;

            return PartialView("_QueryResults", result.ToPagedList(pageNumber, PageList_PageSize));
        }


        //
        // Query raw data methods
        //

        [HttpPost]
        public virtual ActionResult QueryRawData(FormCollection collection)
        {
            query = QueryType.RawDataQuery;

            IEnumerable<CCS_RawData_V1> result = FilterApplied<CCS_RawData_V1>(collection);

            int pageNumber = 1;

            return PartialView("_QueryResults1", result.ToPagedList(pageNumber, PageList_PageSize));
        }

        public virtual ActionResult QueryRawDataPaged(int? page)
        {
            query = QueryType.RawDataQuery;

            IEnumerable<CCS_RawData_V1> result = (IEnumerable<CCS_RawData_V1>)Session["filterResults"];

            int pageNumber = page ?? 1;

            return PartialView("_QueryResults1", result.ToPagedList(pageNumber, PageList_PageSize));
        }


        public virtual IEnumerable<T> GetQueryResult<T>(FormCollection collection, List<AbstractFilter> searchCriteria, QueryType queryType) where T : class
        {
            switch (queryType)
            {
                case QueryType.MeterProfileQuery:
                    return GetMeterProfileQueryResult<T>(collection, searchCriteria);
                case QueryType.RawDataQuery:
                    return GetRawDataQueryResult<T>(collection, searchCriteria);
                case QueryType.BillQuery:
                    return GetBillQueryResult<T>(collection, searchCriteria);
                default:
                    throw new Exception("Unsupported query type: " + queryType.ToString());
            }
        }

        public IEnumerable<T> GetMeterProfileQueryResult<T>(FormCollection collection, List<AbstractFilter> searchCriteria) where T : class
        {
            // Open database
            RawDataContext db = (RawDataContext)RawDataContextFactory.GetInstance().GetDBEFContext();

            var TType = typeof(T);
            if (TType != typeof(CCS_MeterData_V1))
            {
                return new List<T>();
            }

            IQueryable<CCS_MeterData_V1> meters;
            IEnumerable<CCS_MeterData_V1> result = null;

            try
            {
                meters = (from m in db.MeterDataTable
                          select m).AsQueryable().ApplySearchCriteria(searchCriteria.AsEnumerable());

            }
            catch (Exception e)
            {
                throw new HttpException(500, e.Message);
            }


            if (meters == null)
            {
                result = new List<CCS_MeterData_V1>();
            }
            else
            {
                result = meters.ToList();
            }

            return result as IEnumerable<T>;
        }

        public IEnumerable<T> GetRawDataQueryResult<T>(FormCollection collection, List<AbstractFilter> searchCriteria) where T : class
        {
            // Open database
            RawDataContext db = (RawDataContext)RawDataContextFactory.GetInstance().GetDBEFContext();

            var TType = typeof(T);
            if (TType != typeof(CCS_RawData_V1))
            {
                return new List<T>();
            }

            IQueryable<CCS_RawData_V1> data;
            IEnumerable<CCS_RawData_V1> result = null;

            try
            {
                data = (from d in db.RawDataTable
                        select d).AsQueryable().ApplySearchCriteria(searchCriteria.AsEnumerable());

            }
            catch (Exception e)
            {
                throw new HttpException(500, e.Message);
            }


            if (data == null)
            {
                result = new List<CCS_RawData_V1>();
            }
            else
            {
                result = data.ToList();
            }

            return result as IEnumerable<T>;

        }


        public IEnumerable<T> GetBillQueryResult<T>(FormCollection collection, List<AbstractFilter> searchCriteria) where T : class
        {

            string billType = collection["billType"] as string;

            // Open database
            BillContext db = (BillContext)BillContextFactory.GetInstance().GetDBEFContext();

            var TType = typeof(T);
            if (TType != typeof(CCS_Bill_Joined_V1))
            {
                return new List<T>();
            }
            //IEnumerable<Bill> bills = db.Bills.AsQueryable().ApplySearchCriteria(searchCriterias.AsEnumerable());
            //IQueryable<Bill> bills = db.Bills.AsQueryable().ApplySearchCriteria(searchCriteria.AsEnumerable()).OrderBy(s => s.BillID);

            IQueryable<CCS_Bill_Joined_V1> bills;
            IEnumerable<CCS_Bill_Joined_V1> result = null;

            try
            {

                bills = (from b in db.BillTable
                         join u in db.UserTable on b.UserID equals u.UserID
                         where b.BillType == billType
                         orderby b.BillID
                         select new CCS_Bill_Joined_V1()
                         {
                             BillID = b.BillID,
                             AmountDue = b.AmountDue,
                             BillType = b.BillType,
                             CustomerID = b.CustomerID,
                             UserID = b.UserID,
                             FirstName = u.FirstName,
                             MiddleName = u.MiddleName,
                             LastName = u.LastName,
                             Cellphone = u.Cellphone,
                             SecondPhone = u.SecondPhone,
                             Email = u.Email,
                             Address = u.Address,
                             ModifiedDate = u.ModifiedDate
                         }).AsQueryable().ApplySearchCriteria(searchCriteria.AsEnumerable());

            }
            catch (Exception e)
            {
                throw new HttpException(500, e.Message);
            }


            if (bills == null)
            {
                result = new List<CCS_Bill_Joined_V1>();
            }
            else
            {
                result = bills.ToList();
            }

            return result as IEnumerable<T>;
        }



        #endregion


        //Download bills
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<ActionResult> DownloadQueryResult(string queryType)
        {
            query = Utility.ParseEnum<QueryType>(queryType);

            if (query != QueryType.RawDataQuery && query != QueryType.BillQuery)
            {
                throw new Exception(string.Format("Download for {0} is not support", query.ToString()));
            }

            if (Session["filterCriteria"] == null)
            {
                throw new Exception("Session[\"filterCriteria\"] is null");
            }

            string billType = (string)Session["BillType"];

            List<AbstractFilter> filterCriteria = (List<AbstractFilter>)Session["filterCriteria"];

            // Add a message to message queue
            string messageId = Guid.NewGuid().ToString();

            FilterCriteriaMsg<AbstractFilter> filterCriteriaMsg = new FilterCriteriaMsg<AbstractFilter>(messageId, queryType, billType, filterCriteria);

            byte[] msg = filterCriteriaMsg.SerializeFilterCriterias();

            CloudQueueMessage queueMsg = new CloudQueueMessage(msg);
            await queryInitiateQueue.AddMessageAsync(queueMsg);

            // Wait for blob to be ready for downloading
            while (true)
            {
                CloudQueueMessage response;
                response = queryDownloadQueue.GetMessage();
                try
                {
                    if (response != null)
                    {
                        return await ProcessQueueMessage(response);
                    }
                    else
                    {
                        System.Threading.Thread.Sleep(1000);
                    }
                }
                catch (StorageException e)
                {
                    Trace.TraceError("StorageException in DownloadQueryResult: '{0}'", e.Message);
                    if (response != null && response.DequeueCount > 5)
                    {
                        queryDownloadQueue.DeleteMessage(response);
                    }
                    System.Threading.Thread.Sleep(5000);
                }
                catch (Exception e)
                {
                    Trace.TraceError("Exception in DownloadQueryResult: '{0}'", e.Message);
                    break;
                }
            }

            return View();
        }


        protected async Task<ActionResult> ProcessQueueMessage(CloudQueueMessage msg)
        {
            Trace.TraceInformation("Processing queue message '{0}'", msg);

            // Remove message from the queue
            queryDownloadQueue.DeleteMessage(msg);
            string blobName = msg.AsString;

            try
            {
                //Uri blobUri = new Uri(msg.ToString());
                //CloudBlockBlob blob = new CloudBlockBlob(blobUri);

                // Open storage account
                CloudStorageAccount storageAccount = CloudStorageAccount.Parse(RoleEnvironment.GetConfigurationSettingValue("StorageConnectionString"));

                // Get reference to queryDownloadBlob
                CloudBlobClient blobClient = storageAccount.CreateCloudBlobClient();
                queryBlobContainer = blobClient.GetContainerReference("query");
                CloudBlockBlob blob = queryBlobContainer.GetBlockBlobReference(blobName);

                using (MemoryStream memStream = new MemoryStream())
                {
                    await blob.DownloadToStreamAsync(memStream);

                    return new FileContentResult(memStream.ToArray(), blob.Properties.ContentType)
                    {
                        FileDownloadName = blobName + ".csv"
                    };
                }
            }
            catch (Exception)
            {
                return new HttpNotFoundResult();
            }

        }

    }
}
