﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.WindowsAzure;
using Microsoft.WindowsAzure.Diagnostics;
using Microsoft.WindowsAzure.ServiceRuntime;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.Storage.Queue;
using Microsoft.WindowsAzure.Storage.Blob;
using CCSPrototypeCommon;
using CCSPrototypeCommon.Query;
using Microsoft.WindowsAzure.Storage.RetryPolicies;
using System.IO;
using System.Web.Mvc;
using System.Text;
using System.Globalization;
using ThreadedWorkerRole;
using Microsoft.Azure;

namespace HomeWorker
{
    public class BuildQueryBlobWorker : WorkerEntryPoint
    {
        private readonly CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private readonly ManualResetEvent runCompleteEvent = new ManualResetEvent(false);

        private CloudQueue queryDownloadQueue;
        private CloudQueue queryInitiateQueue;
        private static CloudBlobContainer queryBlobContainer;

        const int PageSize = 300;
        const string QueryBlobPrefix = "DownloadQueryResult";

        public override void Run()
        {
            Trace.TraceInformation("WorkerRoleBuildQueryBlob is running");

            CloudQueueMessage msg = null;

            while (true)
            {
                try
                {
                    msg = this.queryInitiateQueue.GetMessage();
                    if (msg != null)
                    {
                        // Remove message from the queue
                        this.queryInitiateQueue.DeleteMessage(msg);
                        ProcessQueueMessage(msg);
                    }
                    else
                    {
                        System.Threading.Thread.Sleep(1000);
                    }
                }
                catch (StorageException e)
                {
                    Trace.TraceError("StorageException in WorkerRoleBuildQueryBlob: '{0}'", e.Message);
                    logException(e);
                    if (msg != null && msg.DequeueCount > 5)
                    {
                        this.queryInitiateQueue.DeleteMessage(msg);
                    }

                    System.Threading.Thread.Sleep(5000);
                }
                catch (Exception e)
                {
                    Trace.TraceError("Exception in WorkerRoleBuildQueryBlob: '{0}'", e.Message);
                    logException(e);
                }

            }

        }

        private async void ProcessQueueMessage(CloudQueueMessage msg)
        {
            Trace.TraceInformation("Processing queue message '{0}'", msg);

            FilterCriteriaMsg<AbstractFilter> filterMsg = msg.DeserializeFilterCriteria<AbstractFilter>();
            string queryType = filterMsg.queryType;

            QueryType query = Utility.ParseEnum<QueryType>(queryType);

            switch (query)
            {
                case QueryType.RawDataQuery:
                    ProcessRawDataQueryDownload(filterMsg);
                    break;
                case QueryType.BillQuery:
                    ProcessBillQueryDownload(filterMsg);
                    break;
                default:
                    throw new Exception("Unsupported query type: " + query.ToString());
            }

        }

        private async void ProcessRawDataQueryDownload(FilterCriteriaMsg<AbstractFilter> filterMsg)
        {
            List<AbstractFilter> filterCriteria = filterMsg.filterCriteria;

            // Open database
            var db = (RawDataContext)RawDataContextFactory.GetInstance().GetDBEFContext();

            // Query database
            IQueryable<CCS_RawData_V1> result =
                            (from m in db.RawDataTable select m).AsQueryable().ApplySearchCriteria(filterCriteria.AsEnumerable()).OrderBy(x => x.MeterID);

            int? page = 0;
            long blockID = 0;
            int pages = (result.Count() + PageSize - 1) / PageSize;
            string queryBlobName = String.Format("{0}-{1}", QueryBlobPrefix, filterMsg.msgID);
            CloudBlockBlob blob = queryBlobContainer.GetBlockBlobReference(queryBlobName);
            Uri blobUri = blob.Uri;

            List<string> blockIDList = new List<string>();

            while (page < pages)
            {

                IEnumerable<CCS_RawData_V1> paginatedData = result.Skip((page ?? 0) * PageSize).Take(PageSize).ToList();

                List<byte> blobBytes = new List<byte>();
                foreach (var item in paginatedData)
                {
                    StringBuilder sb = new StringBuilder();

                    sb.Append(item.MeasureType).Append(",");
                    sb.Append(item.MeterID).Append(",");
                    sb.Append(item.MeterType).Append(",");
                    sb.Append(item.MeterModel).Append(",");
                    sb.Append(item.Manufacturer).Append(",");
                    sb.Append(item.RateSchedule).Append(",");
                    sb.Append(item.ReadType).Append(",");
                    sb.Append(item.Office).Append(",");
                    sb.Append(item.MeterAddress).Append(",");
                    sb.Append(item.Read).Append(",");
                    sb.Append(item.Factor).Append(",");
                    sb.Append(item.Consumption).Append(",");
                    sb.Append(item.Status).Append(",");
                    sb.Append(item.UpdateTimeStamp).Append("\r\n");

                    // Convert string builder to stream
                    blobBytes.AddRange(Encoding.UTF8.GetBytes(sb.ToString()));
                }

                using (MemoryStream ms = new MemoryStream(blobBytes.ToArray()))
                {
                    // Convert blockID to Base64 string
                    var blockIDBase64 = Convert.ToBase64String(Encoding.UTF8.GetBytes(blockID.ToString(CultureInfo.InvariantCulture).PadLeft(32, '0')));
                    blockIDList.Add(blockIDBase64);
                    blob.PutBlock(blockIDBase64, ms, null);
                    ++blockID;
                }

                ++page;
            }

            // Merge block list
            blob.PutBlockList(blockIDList);

            // Add a message to message queue
            //CloudQueueMessage queueMsg = new CloudQueueMessage(blobUri.ToString());
            CloudQueueMessage queueMsg = new CloudQueueMessage(queryBlobName);
            await queryDownloadQueue.AddMessageAsync(queueMsg);

        }

        private async void ProcessBillQueryDownload(FilterCriteriaMsg<AbstractFilter> filterMsg)
        {
            string billType = filterMsg.subType;
            List<AbstractFilter> filterCriteria = filterMsg.filterCriteria;

            // Open database
            var db = (BillContext)BillContextFactory.GetInstance().GetDBEFContext();

            // Query database
            IQueryable<CCS_Bill_Joined_V1> result =
                            (from b in db.BillTable
                             join u in db.UserTable on b.UserID equals u.UserID
                             where b.BillType == billType
                             orderby b.BillID
                             select new CCS_Bill_Joined_V1()
                             {
                                 BillID = b.BillID,
                                 AmountDue = b.AmountDue,
                                 BillType = b.BillType,
                                 CustomerID = b.CustomerID,
                                 UserID = b.UserID,
                                 FirstName = u.FirstName,
                                 MiddleName = u.MiddleName,
                                 LastName = u.LastName,
                                 Cellphone = u.Cellphone,
                                 SecondPhone = u.SecondPhone,
                                 Email = u.Email,
                                 Address = u.Address,
                                 ModifiedDate = u.ModifiedDate
                             }).AsQueryable().ApplySearchCriteria(filterCriteria.AsEnumerable()).OrderBy(x => x.BillID);

            int? page = 0;
            long blockID = 0;
            int pages = (result.Count() + PageSize - 1) / PageSize;
            string queryBlobName = String.Format("{0}-{1}", QueryBlobPrefix, filterMsg.msgID);
            CloudBlockBlob blob = queryBlobContainer.GetBlockBlobReference(queryBlobName);
            Uri blobUri = blob.Uri;

            List<string> blockIDList = new List<string>();

            while (page < pages)
            {

                IEnumerable<CCS_Bill_Joined_V1> paginatedData = result.Skip((page ?? 0) * PageSize).Take(PageSize).ToList();

                List<byte> blobBytes = new List<byte>();
                foreach (var item in paginatedData)
                {
                    StringBuilder sb = new StringBuilder();
                    sb.Append(item.BillID).Append(",");
                    sb.Append(item.BillType).Append(",");
                    sb.Append(item.CustomerID).Append(",");
                    sb.Append(item.UserID).Append(",");
                    sb.Append(item.AmountDue).Append(",");
                    sb.Append(item.FirstName).Append(",");
                    sb.Append(item.LastName).Append(",");
                    sb.Append(item.MiddleName).Append(",");
                    sb.Append(item.Cellphone).Append(",");
                    sb.Append(item.SecondPhone).Append(",");
                    sb.Append(item.Email).Append(",");
                    sb.Append(item.Address).Append("\r\n");

                    // Convert string builder to stream
                    blobBytes.AddRange(Encoding.UTF8.GetBytes(sb.ToString()));
                }

                using (MemoryStream ms = new MemoryStream(blobBytes.ToArray()))
                {
                    // Convert blockID to Base64 string
                    var blockIDBase64 = Convert.ToBase64String(Encoding.UTF8.GetBytes(blockID.ToString(CultureInfo.InvariantCulture).PadLeft(32, '0')));
                    blockIDList.Add(blockIDBase64);
                    blob.PutBlock(blockIDBase64, ms, null);
                    ++blockID;
                }

                ++page;
            }

            // Merge block list
            blob.PutBlockList(blockIDList);

            // Add a message to message queue
            //CloudQueueMessage queueMsg = new CloudQueueMessage(blobUri.ToString());
            CloudQueueMessage queueMsg = new CloudQueueMessage(queryBlobName);
            await queryDownloadQueue.AddMessageAsync(queueMsg);

        }


        public override bool OnStart()
        {
            // Set the maximum number of concurrent connections
            ServicePointManager.DefaultConnectionLimit = 12;

            // For information on handling configuration changes
            // see the MSDN topic at http://go.microsoft.com/fwlink/?LinkId=166357.

            bool result = base.OnStart();

            Trace.TraceInformation("WorkerRoleBuildQueryBlob has been started");

            // Open storage account
            CloudStorageAccount storageAccount = CloudStorageAccount.Parse(RoleEnvironment.GetConfigurationSettingValue("StorageConnectionString"));

            // Create blob container
            CloudBlobClient blobClient = storageAccount.CreateCloudBlobClient();
            queryBlobContainer = blobClient.GetContainerReference("query");
            if (queryBlobContainer.CreateIfNotExists())
            {
                queryBlobContainer.SetPermissions(
                    new BlobContainerPermissions
                    {
                        PublicAccess = BlobContainerPublicAccessType.Blob
                    });
            }

            // Create quere storage
            CloudQueueClient queueClient = storageAccount.CreateCloudQueueClient();
            queryInitiateQueue = queueClient.GetQueueReference("queryinit");
            queryDownloadQueue = queueClient.GetQueueReference("querydownload");
            queryInitiateQueue.CreateIfNotExists();
            queryDownloadQueue.CreateIfNotExists();

            Trace.TraceInformation("Storage initialized");

            return result;
        }

        public override void OnStop()
        {
            Trace.TraceInformation("WorkerRoleBuildQueryBlob is stopping");

            this.cancellationTokenSource.Cancel();
            this.runCompleteEvent.WaitOne();

            base.OnStop();

            Trace.TraceInformation("WorkerRoleBuildQueryBlob has stopped");
        }

        private async Task RunAsync(CancellationToken cancellationToken)
        {
            // TODO: Replace the following with your own logic.
            while (!cancellationToken.IsCancellationRequested)
            {
                Trace.TraceInformation("Working");
                await Task.Delay(1000);
            }
        }
    }
}
