﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Diagnostics;
using System.Configuration;
using System.Linq;
using System.Net;
using System.Threading;
using System.Threading.Tasks;
using System.Data.Entity;
using System.Data;
using System.Web;
using CCSPrototypeCommon;
using Microsoft.WindowsAzure;
using Microsoft.WindowsAzure.Diagnostics;
using Microsoft.WindowsAzure.ServiceRuntime;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.Storage.Blob;
using Microsoft.WindowsAzure.Storage.Queue;
using Microsoft.WindowsAzure.Storage.Table;
using Microsoft.WindowsAzure.Storage.Table.DataServices;
using Microsoft.WindowsAzure.Storage.RetryPolicies;
using ThreadedWorkerRole;
using Microsoft.Azure;
using Newtonsoft.Json;

namespace HomeWorker
{
    public class TransferBillToDBWorker : WorkerEntryPoint
    {
        private CloudQueue filesQueue;
        private CloudBlobContainer filesBlobContainer;
        private CloudTable _templateTable;

        public override void Run()
        {
            Trace.TraceInformation("TransferBillToDBWorker is running");

            CloudQueueMessage msg = null;
            while (true)
            {
                try
                {
                    msg = this.filesQueue.GetMessage();

                    if (msg != null)
                    {
                        filesQueue.DeleteMessage(msg);
                        ProcessQueueMessage(msg);
                    }
                    else
                    {
                        System.Threading.Thread.Sleep(1000);
                    }
                }
                catch (StorageException e)
                {
                    Trace.TraceError("StorageException in TransferBillToDBWorker: '{0}'", e.Message);
                    logException(e);
                    if (msg != null && msg.DequeueCount > 5)
                    {
                        this.filesQueue.DeleteMessage(msg);
                    }

                    System.Threading.Thread.Sleep(5000);
                }
                catch (Exception e)
                {
                    Trace.TraceError("Exception in TransferBillToDBWorker: '{0}'", e.Message);
                    logException(e);
                }

            }
        }


        private async void ProcessQueueMessage(CloudQueueMessage msg)
        {
            Trace.TraceInformation("Processing queue message '{0}'", msg);

            var db = (BillContext)BillContextFactory.GetInstance().GetDBEFContext();

            BillQueueMsg billMsg = await Task.Factory.StartNew(() => JsonConvert.DeserializeObject<BillQueueMsg>(msg.AsString));
            if (string.IsNullOrWhiteSpace(billMsg.BlobName))
            {
                throw new Exception("Null blob name");
            }
            string customerID = Utility.GetCustomerIDFromBlobName(billMsg.BlobName);
            // Insert records in DB
            try
            {
                // Create a retrieve operation that takes a customer entity.
                TableOperation retrieveOperation = TableOperation.Retrieve<BillTemplateTableEntity>(customerID, billMsg.BillType);
                // Execute the retrieve operation.
                TableResult retrievedResult = _templateTable.Execute(retrieveOperation);
                BillTemplateTableEntity billTemplate = null;
                if (retrievedResult != null && retrievedResult.Result != null)
                {
                    billTemplate = retrievedResult.Result as BillTemplateTableEntity;
                }
                else
                {
                    throw new BillDataException("No bill template");
                }

                CloudBlockBlob fileBlob = this.filesBlobContainer.GetBlockBlobReference(billMsg.BlobName);
                using (Stream fs = fileBlob.OpenRead())
                {
                    StreamReader csv = new StreamReader(fs);
                    // Consume header
                    var line = csv.ReadLine();
                    string[] headerList = line.Split(',');
                    var billTemplateMapping = Utility.GetColumnMappingByHeader(new List<string>(headerList), billTemplate.GetColumnList());
                    while ((line = csv.ReadLine()) != null)
                    {
                        Trace.TraceInformation("line {0}", line);
                        string[] columnList = line.Split(',');
                        foreach (string column in columnList)
                        {
                            Trace.TraceInformation("str {0}", column);
                        }
                        var billRecord = new CCS_Bill_V1();
                        var userRecord = new CCS_User_V1();
                        
                        if (billTemplate != null)
                        {
                            billRecord.AmountDue = Double.Parse(columnList[billTemplateMapping[billTemplate.AmountDueColumn]]);
                            billRecord.BillType = billMsg.BillType;
                            billRecord.CustomerID = Guid.Parse(customerID);
                            userRecord.FirstName = columnList[billTemplateMapping[billTemplate.UserNameColumn]];
                            userRecord.Cellphone = columnList[billTemplateMapping[billTemplate.CellphoneColumn]];
                            userRecord.Address = columnList[billTemplateMapping[billTemplate.UserAddressColumn]];
                            try
                            {
                                var IDNumberHeaders = new List<string> ( new string[]
                                {
                                    "身份证", "身份证号"
                                });
                                var IDNumberColumn = Utility.GuessColumnByHeaderNames(new List<string>(headerList), IDNumberHeaders);
                                userRecord.IDNumber = columnList[IDNumberColumn];
                            }
                            catch (BillDataException e)
                            {
                                // This is expected in some case
                                logException(e);
                            }
                            for (var index = 0; index < headerList.Length; ++index)
                            {
                                if (string.IsNullOrWhiteSpace(headerList[index]))
                                {
                                    continue;
                                }
                                var billInfoRecord = new CCS_BillInfo_V1();
                                billInfoRecord.Name = headerList[index];
                                billInfoRecord.Value = columnList[index];
                                db.BillInfoTable.Add(billInfoRecord);

                                string headerColumn = headerList[index];
                                try
                                {
                                    if (!db.BillTemplateTable.Any(obj => obj.BillType == billMsg.BillType && obj.Column == headerColumn))
                                    {
                                        var billTemplateRow = new CCS_BillTemplate_V1();
                                        billTemplateRow.BillType = billMsg.BillType;
                                        billTemplateRow.CustomerID = billRecord.CustomerID;
                                        billTemplateRow.Column = headerColumn;
                                        billTemplateRow.ColumnType = ParseValue(columnList[index]);
                                        db.BillTemplateTable.Add(billTemplateRow);
                                    }
                                }
                                catch (Exception e)
                                { 
                                    // Ignore exceptions
                                    logException(e);
                                }
                            }

                            // to do bill info table
                            //bill.UserID = dataTemplate.ContainsKey("UserID") ? columns[(int)dataTemplate["UserID"].Int32Value] : "defaultUserID";
                            //bill.Address = dataTemplate.ContainsKey("Address") ? columns[(int)dataTemplate["Address"].Int32Value] : "defaultAddress";
                            //bill.MobilePhone = dataTemplate.ContainsKey("MobilePhone") ? columns[(int)dataTemplate["MobilePhone"].Int32Value] : "defaultMobilePhone";
                            //bill.MeterID = dataTemplate.ContainsKey("MeterID") ? columns[(int)dataTemplate["MeterID"].Int32Value] : "defaultMeterID";
                            //bill.ReadingEnd = dataTemplate.ContainsKey("ReadingEnd") ? Double.Parse(columns[(int)dataTemplate["ReadingEnd"].Int32Value]) : 0d;
                            //bill.WaterUsage = dataTemplate.ContainsKey("WaterUsage") ? Double.Parse(columns[(int)dataTemplate["WaterUsage"].Int32Value]) : 0d;
                            //bill.AmountDue = dataTemplate.ContainsKey("AmountDue") ? Double.Parse(columns[(int)dataTemplate["AmountDue"].Int32Value]) : 0d;
                        }
                        else
                        {
                            throw new BillDataException("No bill template");
                            //bill.UserID = columns[2];
                            //bill.Address = columns[5];
                            //bill.ReadingEnd = Double.Parse(columns[11]);
                            //bill.WaterUsage = Double.Parse(columns[12]);
                            //bill.AmountDue = Double.Parse(columns[22]);
                            //bill.MobilePhone = columns[34];
                            //bill.MeterID = columns[42];
                        }
                        //bill.TimeStamp = DateTime.UtcNow;
                        //db.Bills.Add(bill);

                        billRecord.ModifiedDate = DateTime.UtcNow;
                        userRecord.ModifiedDate = DateTime.UtcNow;
                        db.BillTable.Add(billRecord);
                        db.UserTable.Add(userRecord);
                        await db.SaveChangesAsync();
                    }
                }
            }
            catch (Exception e)
            {
                Trace.TraceError(String.Format("Exception in ProcessQueueMessage: '{0}'", e.Message));
                logException(e);
            }

        }

        private string ParseValue(string value)
        {
            string result;
            Int32 intValue;
            Int64 bigIntValue;
            Double doubleValue;
            DateTime dateValue;

            if (Int32.TryParse(value, out intValue) || Int64.TryParse(value, out bigIntValue))
            {
                result = "Integer";
            }
            else if (Double.TryParse(value, out doubleValue))
            {
                result = "Double";
            }
            else if (DateTime.TryParse(value, out dateValue))
            {
                result = "DateTime";
            }
            else
            {
                result = "String";
            }

            return result;
        }


        public override bool OnStart()
        {
            // Set the maximum number of concurrent connections
            ServicePointManager.DefaultConnectionLimit = 12;

            // For information on handling configuration changes
            // see the MSDN topic at http://go.microsoft.com/fwlink/?LinkId=166357.


            // Open storage account
            CloudStorageAccount storageAccount = CloudStorageAccount.Parse(RoleEnvironment.GetConfigurationSettingValue("StorageConnectionString"));

            // Create blob container
            CloudBlobClient blobClient = storageAccount.CreateCloudBlobClient();
            filesBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.BillStorageName);
            if (filesBlobContainer.CreateIfNotExists())
            {
                filesBlobContainer.SetPermissions(
                    new BlobContainerPermissions
                    {
                        PublicAccess = BlobContainerPublicAccessType.Blob
                    });
            }

            // Create quere storage
            CloudQueueClient queueClient = storageAccount.CreateCloudQueueClient();
            filesQueue = queueClient.GetQueueReference(CCSPrototypeCommon.Constants.BillStorageName);
            filesQueue.CreateIfNotExists();

            // Get table context object
            CloudTableClient tableClient = storageAccount.CreateCloudTableClient();
            tableClient.DefaultRequestOptions.RetryPolicy = new LinearRetry(TimeSpan.FromSeconds(3), 3);
            _templateTable = tableClient.GetTableReference(Constants.DataTemplateTableName);
            _templateTable.CreateIfNotExists();

            Trace.TraceInformation("Storage initialized");

            return base.OnStart();
        }

    }
}
