﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Diagnostics;
using System.Configuration;
using System.Linq;
using System.Net;
using System.Threading;
using System.Threading.Tasks;
using System.Data.Entity;
using System.Data;
using System.Web;
using CCSPrototypeCommon;
using Microsoft.WindowsAzure;
using Microsoft.WindowsAzure.Diagnostics;
using Microsoft.WindowsAzure.ServiceRuntime;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.Storage.Blob;
using Microsoft.WindowsAzure.Storage.Queue;
using Microsoft.WindowsAzure.Storage.Table;
using Microsoft.WindowsAzure.Storage.Table.DataServices;
using Microsoft.WindowsAzure.Storage.RetryPolicies;
using ThreadedWorkerRole;
using Microsoft.Azure;
using Newtonsoft.Json;
using System.Globalization;

namespace HomeWorker
{
    public class TransferMeterProfileToDBWorker : WorkerEntryPoint
    {
        private CloudQueue filesQueue;
        private CloudBlobContainer filesBlobContainer;

        public override void Run()
        {
            Trace.TraceInformation("TransferMeterProfileToDBWorker is running");

            CloudQueueMessage msg = null;

            while (true)
            {
                try
                {
                    msg = this.filesQueue.GetMessage();

                    if (msg != null)
                    {
                        filesQueue.DeleteMessage(msg);
                        ProcessQueueMessage(msg);
                    }
                    else
                    {
                        System.Threading.Thread.Sleep(1000);
                    }
                }
                catch (StorageException e)
                {
                    Trace.TraceError("StorageException in TransferMeterProfileToDBWorker: '{0}'", e.Message);
                    logException(e);
                    if (msg != null && msg.DequeueCount > 5)
                    {
                        this.filesQueue.DeleteMessage(msg);
                    }

                    System.Threading.Thread.Sleep(5000);
                }
                catch (Exception e)
                {
                    Trace.TraceError("Exception in TransferMeterProfileToDBWorker: '{0}'", e.Message);
                    logException(e);
                }

            }
        }


        private async void ProcessQueueMessage(CloudQueueMessage msg)
        {
            Trace.TraceInformation("Processing queue message '{0}'", msg);

            var db = (RawDataContext)RawDataContextFactory.GetInstance().GetDBEFContext();

            MeterProfileQueueMsg meterProfileMsg = await Task.Factory.StartNew(() => JsonConvert.DeserializeObject<MeterProfileQueueMsg>(msg.AsString));
            string blobName = meterProfileMsg.BlobName;
            if (string.IsNullOrWhiteSpace(blobName))
            {
                throw new Exception("Null blob name");
            }

            // Insert records in DB
            try
            {
                CloudBlockBlob fileBlob = this.filesBlobContainer.GetBlockBlobReference(blobName);
                using (Stream fs = fileBlob.OpenRead())
                {
                    StreamReader csv = new StreamReader(fs);
                    // Consume header
                    var line = csv.ReadLine();
                    while ((line = csv.ReadLine()) != null)
                    {
                        Trace.TraceInformation("line {0}", line);
                        string[] columnList = line.Split(',');
                        foreach (string column in columnList)
                        {
                            Trace.TraceInformation("str {0}", column);
                        }

                        try
                        {
                            string operation = columnList[23];
                            string meterID = columnList[5];
                            if (string.IsNullOrEmpty(operation) || String.Compare(operation, "A", true) == 0)
                            {
                                // Add record
                                InsertRecord(db, columnList);
                            }
                            else if (String.Compare(operation, "M", true) == 0)
                            {
                                // Update record
                                if (meterID != null)
                                {
                                    var oldRecord = db.MeterDataTable.Where(r => r.MeterID == meterID && r.ModifiedDate == null).FirstOrDefault();

                                    if (oldRecord != null)
                                    {
                                        oldRecord.ModifiedDate = DateTime.UtcNow;
                                    }
                                }

                                InsertRecord(db, columnList);
                            }
                            else if (String.Compare(columnList[23], "R", true) == 0)
                            {
                                // Remove record
                                if (meterID != null)
                                {
                                    var oldRecord = db.MeterDataTable.Where(r => r.MeterID == meterID && r.ModifiedDate == null).FirstOrDefault();

                                    if (oldRecord != null)
                                    {
                                        oldRecord.ModifiedDate = DateTime.UtcNow;
                                    }
                                }
                            }
                            else
                            {
                                // Invalid option
                                throw new Exception(String.Format("Invalid operation type: {0}", operation));
                            }
                        }
                        catch (Exception e)
                        {
                            throw new Exception(String.Format("Insert meter data into table failed with exception: {0}", e.Message));
                        }

                        await db.SaveChangesAsync();
                    }
                }
            }
            catch (Exception e)
            {
                Trace.TraceError(String.Format("Exception in ProcessQueueMessage: '{0}'", e.Message));
                logException(e);
            }

        }


        private void InsertRecord(RawDataContext db, string[] columnList)
        {
            CCS_MeterData_V1 meterProfileRecord = new CCS_MeterData_V1(); 
            meterProfileRecord.MeterType = columnList[0];
            meterProfileRecord.UserName = columnList[1];
            meterProfileRecord.AccountNumber = columnList[2];
            meterProfileRecord.UserNumber = columnList[3];
            meterProfileRecord.MeterSN = columnList[4];
            meterProfileRecord.MeterID = columnList[5];
            meterProfileRecord.MeterNumber = columnList[6];
            meterProfileRecord.Address = columnList[7];
            meterProfileRecord.ApartmentNumber = columnList[8];
            meterProfileRecord.UserType = columnList[9];
            meterProfileRecord.Office = columnList[10];
            meterProfileRecord.PhoneNumber = columnList[11];
            meterProfileRecord.Memo1 = columnList[12];
            meterProfileRecord.Memo2 = columnList[13];
            CultureInfo provider = CultureInfo.InvariantCulture;
            meterProfileRecord.CutoffDate = DateTime.ParseExact(columnList[14], "MM/dd/yyyy", provider);
            meterProfileRecord.CollectType = columnList[15];
            meterProfileRecord.TerminalNumber = columnList[16];
            meterProfileRecord.TerminalEquipmentID = columnList[17];
            meterProfileRecord.TerminalAddress = columnList[18];
            meterProfileRecord.Latitude = Double.Parse(columnList[19]);
            meterProfileRecord.Longitude = Double.Parse(columnList[20]);
            meterProfileRecord.CreationDate = DateTime.UtcNow;

            db.MeterDataTable.Add(meterProfileRecord);

        }


        public override bool OnStart()
        {
            // Set the maximum number of concurrent connections
            ServicePointManager.DefaultConnectionLimit = 12;

            // For information on handling configuration changes
            // see the MSDN topic at http://go.microsoft.com/fwlink/?LinkId=166357.


            // Open storage account
            CloudStorageAccount storageAccount = CloudStorageAccount.Parse(RoleEnvironment.GetConfigurationSettingValue("StorageConnectionString"));

            // Create blob container
            CloudBlobClient blobClient = storageAccount.CreateCloudBlobClient();
            filesBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.MeterProfileStorageName);
            if (filesBlobContainer.CreateIfNotExists())
            {
                filesBlobContainer.SetPermissions(
                    new BlobContainerPermissions
                    {
                        PublicAccess = BlobContainerPublicAccessType.Blob
                    });
            }

            // Create quere storage
            CloudQueueClient queueClient = storageAccount.CreateCloudQueueClient();
            filesQueue = queueClient.GetQueueReference(CCSPrototypeCommon.Constants.MeterProfileStorageName);
            filesQueue.CreateIfNotExists();

            Trace.TraceInformation("Storage initialized");

            return base.OnStart();
        }

    }
}
