﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Diagnostics;
using System.Configuration;
using System.Linq;
using System.Net;
using System.Threading;
using System.Threading.Tasks;
using System.Data.Entity;
using System.Data;
using System.Web;
using CCSPrototypeCommon;
using Microsoft.WindowsAzure;
using Microsoft.WindowsAzure.Diagnostics;
using Microsoft.WindowsAzure.ServiceRuntime;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.Storage.Blob;
using Microsoft.WindowsAzure.Storage.Queue;
using Microsoft.WindowsAzure.Storage.Table;
using Microsoft.WindowsAzure.Storage.Table.DataServices;
using Microsoft.WindowsAzure.Storage.RetryPolicies;
using ThreadedWorkerRole;
using Microsoft.Azure;
using Newtonsoft.Json;
using System.Globalization;

namespace HomeWorker
{
    public class TransferRawDataToDBWorker : WorkerEntryPoint
    {
        private CloudQueue filesQueue;
        private CloudBlobContainer filesBlobContainer;

        public override void Run()
        {
            Trace.TraceInformation("TransferRawDataToDBWorker is running");

            CloudQueueMessage msg = null;
            while (true)
            {
                try
                {
                    msg = this.filesQueue.GetMessage();

                    if (msg != null)
                    {
                        filesQueue.DeleteMessage(msg);
                        ProcessQueueMessage(msg);
                    }
                    else
                    {
                        System.Threading.Thread.Sleep(1000);
                    }
                }
                catch (StorageException e)
                {
                    Trace.TraceError("StorageException in TransferRawDataToDBWorker: '{0}'", e.Message);
                    logException(e);
                    if (msg != null && msg.DequeueCount > 5)
                    {
                        this.filesQueue.DeleteMessage(msg);
                    }

                    System.Threading.Thread.Sleep(5000);
                }
                catch (Exception e)
                {
                    Trace.TraceError("Exception in TransferRawDataToDBWorker: '{0}'", e.Message);

                    logException(e);
                    
                }

            }
        }


        private async void ProcessQueueMessage(CloudQueueMessage msg)
        {
            Trace.TraceInformation("Processing queue message '{0}'", msg);

            var db = (RawDataContext)RawDataContextFactory.GetInstance().GetDBEFContext();

            RawDataQueueMsg rawDataMsg = await Task.Factory.StartNew(() => JsonConvert.DeserializeObject<RawDataQueueMsg>(msg.AsString));
            string blobName = rawDataMsg.BlobName;
            //string blobName = msg.AsString;
            if (string.IsNullOrWhiteSpace(blobName))
            {
                throw new Exception("Null blob name");
            }

            // Insert records in DB
            try
            {
                CloudBlockBlob fileBlob = this.filesBlobContainer.GetBlockBlobReference(blobName);
                using (Stream fs = fileBlob.OpenRead())
                {
                    StreamReader csv = new StreamReader(fs);
                    // Consume header
                    var line = csv.ReadLine();
                    while ((line = csv.ReadLine()) != null)
                    {
                        Trace.TraceInformation("line {0}", line);
                        string[] columnList = line.Split(',');
                        foreach (string column in columnList)
                        {
                            Trace.TraceInformation("str {0}", column);
                        }

                        try
                        {
                            var rawDataRecord = new CCS_RawData_V1();

                            rawDataRecord.MeasureType = removeFirstLastDoubleQuotes(columnList[0]);
                            rawDataRecord.MeterID = removeFirstLastDoubleQuotes(columnList[1]);
                            rawDataRecord.MeterType = removeFirstLastDoubleQuotes(columnList[2]);
                            rawDataRecord.MeterModel = removeFirstLastDoubleQuotes(columnList[3]);
                            rawDataRecord.Manufacturer = removeFirstLastDoubleQuotes(columnList[4]);
                            rawDataRecord.RateSchedule = removeFirstLastDoubleQuotes(columnList[5]);
                            rawDataRecord.ReadType = removeFirstLastDoubleQuotes(columnList[6]);
                            rawDataRecord.Office = removeFirstLastDoubleQuotes(columnList[7]);
                            rawDataRecord.MeterAddress = removeFirstLastDoubleQuotes(columnList[8]);
                            rawDataRecord.Read = Double.Parse(removeFirstLastDoubleQuotes(columnList[9]));
                            rawDataRecord.Factor = Double.Parse(removeFirstLastDoubleQuotes(columnList[10]));
                            rawDataRecord.Consumption = Double.Parse(removeFirstLastDoubleQuotes(columnList[11]));
                            rawDataRecord.Status = removeFirstLastDoubleQuotes(columnList[12]);
                            CultureInfo provider = CultureInfo.InvariantCulture;
                            rawDataRecord.UpdateTimeStamp = DateTime.ParseExact(removeFirstLastDoubleQuotes(columnList[13]), "d", provider);
                            db.RawDataTable.Add(rawDataRecord);
                        }
                        catch (Exception e)
                        {
                            throw new Exception(String.Format("Insert raw data into table failed with exception: {0}", e.Message));
                        }

                        await db.SaveChangesAsync();
                    }
                }
            }
            catch (Exception e)
            {
                Trace.TraceError(String.Format("Exception in ProcessQueueMessage: '{0}'", e.Message));
                logException(e);
            }

        }

        public string removeFirstLastDoubleQuotes(string str)
        {
            if (string.IsNullOrEmpty(str))
                return str;
            if(str.StartsWith("\""))
                str = str.Substring(1);
            if (str.EndsWith("\""))
                str = str.Substring(0, str.Length - 1);

            return str;
        }
        public override bool OnStart()
        {
            // Set the maximum number of concurrent connections
            ServicePointManager.DefaultConnectionLimit = 12;

            // For information on handling configuration changes
            // see the MSDN topic at http://go.microsoft.com/fwlink/?LinkId=166357.


            // Open storage account
            CloudStorageAccount storageAccount = CloudStorageAccount.Parse(RoleEnvironment.GetConfigurationSettingValue("StorageConnectionString"));

            // Create blob container
            CloudBlobClient blobClient = storageAccount.CreateCloudBlobClient();
            filesBlobContainer = blobClient.GetContainerReference(CCSPrototypeCommon.Constants.RawDataStorageName);
            if (filesBlobContainer.CreateIfNotExists())
            {
                filesBlobContainer.SetPermissions(
                    new BlobContainerPermissions
                    {
                        PublicAccess = BlobContainerPublicAccessType.Blob
                    });
            }

            // Create quere storage
            CloudQueueClient queueClient = storageAccount.CreateCloudQueueClient();
            filesQueue = queueClient.GetQueueReference(CCSPrototypeCommon.Constants.RawDataStorageName);
            filesQueue.CreateIfNotExists();

            Trace.TraceInformation("Storage initialized");

            return base.OnStart();
        }

    }
}
