using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.WindowsAzure;
//using Microsoft.WindowsAzure.Diagnostics;
using Microsoft.WindowsAzure.ServiceRuntime;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.Storage.Queue;
using Microsoft.WindowsAzure.Storage.Blob;
using CCSPrototypeCommon;
using CCSPrototypeCommon.Filters;
using Microsoft.WindowsAzure.Storage.RetryPolicies;
using System.IO;
using System.Web.Mvc;
using System.Text;
using System.Globalization;
using Microsoft.Azure;

namespace WorkerRoleBuildQueryBlob
{
    public class WorkerRole : RoleEntryPoint
    {
        private readonly CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private readonly ManualResetEvent runCompleteEvent = new ManualResetEvent(false);

        private CloudQueue queryDownloadQueue;
        private CloudQueue queryInitiateQueue;
        private static CloudBlobContainer queryBlobContainer;

        const int PageSize = 300;
        const string QueryBlobPrefix = "DownloadQueryResult";

        public override void Run()
        {
            Trace.TraceInformation("WorkerRoleBuildQueryBlob is running");

            CloudQueueMessage msg = null;

            while (true)
            {
                try
                {
                    msg = this.queryInitiateQueue.GetMessage();
                    if (msg != null)
                    {
                        ProcessQueueMessage(msg);
                    }
                    else
                    {
                        System.Threading.Thread.Sleep(1000);
                    }
                }
                catch (StorageException e)
                {
                    Trace.TraceError("StorageException in WorkerRoleBuildQueryBlob: '{0}'", e.Message);
                    if (msg != null && msg.DequeueCount > 5)
                    {
                        this.queryInitiateQueue.DeleteMessage(msg);
                    }

                    System.Threading.Thread.Sleep(5000);
                }
                catch (Exception e)
                {
                    Trace.TraceError("Exception in WorkerRoleBuildQueryBlob: '{0}'", e.Message);
                }

            }

        }

        private async void ProcessQueueMessage(CloudQueueMessage msg)
        {
            Trace.TraceInformation("Processing queue message '{0}'", msg);

            // Remove message from the queue
            this.queryInitiateQueue.DeleteMessage(msg);

            FilterCriteriasMsg<AbstractFilter> filterCriteriaMsg = msg.DeserializeFilterCriteria<AbstractFilter>();
            List<AbstractFilter> filterCriteria = filterCriteriaMsg.filterCriterias;

            // Open database
            string dbConnString = CloudConfigurationManager.GetSetting("BillDbConnectionString");
            BillContext db = new BillContext(dbConnString);
            db.Database.CreateIfNotExists();

            // Query database
            IQueryable<Bill> result = db.Bills.AsQueryable().ApplySearchCriteria(filterCriteria).OrderBy(x => x.BillID);

            int? page = 0;
            long blockID = 0;
            int pages = (result.Count() + PageSize - 1) / PageSize;
            string queryBlobName = String.Format("{0}-{1}", QueryBlobPrefix, filterCriteriaMsg.id);
            CloudBlockBlob blob = queryBlobContainer.GetBlockBlobReference(queryBlobName);
            Uri blobUri = blob.Uri;

            List<string> blockIDList = new List<string>();

            while (page < pages)
            {
                
                IEnumerable<Bill> paginatedData = result.Skip((page ?? 0) * PageSize)
                                              .Take(PageSize)
                                              .ToList();


                List<byte> blobBytes = new List<byte>();
                foreach (var item in paginatedData)
                {
                    StringBuilder sb = new StringBuilder();

                    sb.Append(item.UserID).Append(",");
                    sb.Append(item.Address).Append(",");
                    sb.Append(item.ReadingEnd).Append(",");
                    sb.Append(item.WaterUsage).Append(",");
                    sb.Append(item.AmountDue).Append(",");
                    sb.Append(item.MobilePhone).Append(",");
                    sb.Append(item.MeterID).Append(",");
                    sb.Append(item.PaymentID).Append(",");
                    sb.Append(item.PaymentType).Append(",");
                    sb.Append(item.IDNumber).Append(",");
                    sb.Append(item.Office).Append(",");
                    sb.Append(item.CollectionQuota).Append(",");
                    sb.Append(item.Reserved1).Append(",");
                    sb.Append(item.Reserved2).Append("\r\n");

                    // Convert string builder to stream
                    blobBytes.AddRange(Encoding.UTF8.GetBytes(sb.ToString()));
                }

                using (MemoryStream ms = new MemoryStream(blobBytes.ToArray()))
                {
                    // Convert blockID to Base64 string
                    var blockIDBase64 = Convert.ToBase64String(Encoding.UTF8.GetBytes(blockID.ToString(CultureInfo.InvariantCulture).PadLeft(32, '0')));
                    blockIDList.Add(blockIDBase64);
                    blob.PutBlock(blockIDBase64, ms, null);
                    ++blockID;
                }

                ++page;
            }

            // Merge block list
            blob.PutBlockList(blockIDList);

            // Add a message to message queue
            //CloudQueueMessage queueMsg = new CloudQueueMessage(blobUri.ToString());
            CloudQueueMessage queueMsg = new CloudQueueMessage(queryBlobName);
            await queryDownloadQueue.AddMessageAsync(queueMsg);

        }


        public override bool OnStart()
        {
            // Set the maximum number of concurrent connections
            ServicePointManager.DefaultConnectionLimit = 12;

            // For information on handling configuration changes
            // see the MSDN topic at http://go.microsoft.com/fwlink/?LinkId=166357.

            bool result = base.OnStart();

            Trace.TraceInformation("WorkerRoleBuildQueryBlob has been started");

            // Open storage account
            CloudStorageAccount storageAccount = CloudStorageAccount.Parse(RoleEnvironment.GetConfigurationSettingValue("StorageConnectionString"));

            // Create blob container
            CloudBlobClient blobClient = storageAccount.CreateCloudBlobClient();
            queryBlobContainer = blobClient.GetContainerReference("query");
            if (queryBlobContainer.CreateIfNotExists())
            {
                queryBlobContainer.SetPermissions(
                    new BlobContainerPermissions
                    {
                        PublicAccess = BlobContainerPublicAccessType.Blob
                    });
            }

            // Create quere storage
            CloudQueueClient queueClient = storageAccount.CreateCloudQueueClient();
            queryInitiateQueue = queueClient.GetQueueReference("queryinit");
            queryDownloadQueue = queueClient.GetQueueReference("querydownload");
            queryInitiateQueue.CreateIfNotExists();
            queryDownloadQueue.CreateIfNotExists();

            Trace.TraceInformation("Storage initialized");

            return result;
        }

        public override void OnStop()
        {
            Trace.TraceInformation("WorkerRoleBuildQueryBlob is stopping");

            this.cancellationTokenSource.Cancel();
            this.runCompleteEvent.WaitOne();

            base.OnStop();

            Trace.TraceInformation("WorkerRoleBuildQueryBlob has stopped");
        }

        private async Task RunAsync(CancellationToken cancellationToken)
        {
            // TODO: Replace the following with your own logic.
            while (!cancellationToken.IsCancellationRequested)
            {
                Trace.TraceInformation("Working");
                await Task.Delay(1000);
            }
        }
    }
}
