<?php

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) exit;

/**
 * Model Class
 *
 * Handles generic plugin functionality, mostly Social Network related
 * for all the different features for the autoposting.
 *
 * @package Social Auto Poster
 * @since 1.0.0
 */
class Wpw_Auto_Poster_Model {

	public function __construct() {
		
	}
	
	/**
	 * similar to checked() but checks for array
	 *	
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */	
	function checked_array( $checked, $current ) {

		if ( is_array( $current ) ) {
            if( in_array( $checked, $current ) ) {
                echo ' checked="checked"';
            }
        } else {
            if( $checked == $current ) {
                echo ' checked="checked"';
            }
        }
	}
	
	/**
	 * Get Unserialize the data
	 * 
	 * Handle serialize data and return unserialize data
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	function get_unserialize_data( $data ) {
	
		$undata = unserialize( $data );
		return $undata;		
	}
	
	/**
	 * Escape Tags & Slashes with URL
	 *
	 * Handles escapping the slashes and tags
	 *
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */   
	function escape_url( $data ) {
	
		return esc_url( $data ); // esc_url will do stripslashes and esc_attr both so we dont need to call it.
	}
	
	/**
	 * Escape Tags & Slashes
	 *
	 * Handles escapping the slashes and tags
	 *
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */   
	public function wpw_auto_poster_escape_attr($data){
		return esc_attr_e(stripslashes($data));
	}
	
	/**
	 * Stripslashes
	 * 
	 * It will stripslashes from the content
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */	
	public function wpw_auto_poster_stripslashes_deep( $data = array(), $flag=false ) {
		
		if($flag != true) {
			$data = $this->wpw_auto_poster_nohtml_kses($data);
		}
		$data = stripslashes_deep($data);
		return $data;
	}
	
	/**
	 * Strip Html Tags 
	 * 
	 * It will sanitize text input (strip html tags, and escape characters)
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	public function wpw_auto_poster_nohtml_kses( $data = array() ) {	
		
		if ( is_array($data) ) {
			
			$data = array_map(array($this,'wpw_auto_poster_nohtml_kses'), $data);
			
		} elseif ( is_string( $data ) ) {
			
			$data = wp_filter_nohtml_kses($data);
		}
		
		return $data;
	}	
	
	/**
	 * HTML Entity Decode
	 * 
	 * Handles to decode HTML entities
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	public function wpw_auto_poster_html_decode( $string ) {
		
		return html_entity_decode( $string );
	}
	/**
	 * Get Shortner Link
	 * 
	 * Handles to return shortner link
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	function wpw_auto_poster_get_short_post_link( $link = '', $unique = 'false', $postid = '', $customlink = 'false', $socialtype = 'fb' ) { //
		
		global $wpw_auto_poster_options;
		
		$shortnertype = $wpw_auto_poster_options[$socialtype.'_url_shortener'];
		
		//default link when it is blank
		$link = !empty( $link ) ? $link : get_permalink( $postid );
		
		if( $unique == 'true' && $customlink == 'true' ) {//unique url && it is custom url
			$link = add_query_arg( 'wpwautoposter', time(), $link );
		}
		
		switch( $shortnertype ) {
			case 'tinyurl':
				require_once( WPW_AUTO_POSTER_DIR . '/includes/shorteners/tinyurl.php' );
				
				$tinyurl = new wpw_auto_poster_tw_tinyurl;
				$link = $tinyurl->shorten( $link );
				break;
			case 'wordpress':
				
				if( $customlink != 'true' ) { //check custom link should not set
					
					if( get_option('permalink_structure') != '' ) {
						$link = wp_get_shortlink( $postid );
					} else {
						$link = get_permalink( $postid );
					}
					
					$link = apply_filters('wpw_custom_permalink', $link, $postid);
					
					if( $unique == 'true') { //unique url
						$link = add_query_arg( 'wpwautoposter', time(), $link );
					}
					
				}
				break;
			case 'bitly':		
				
				require_once( WPW_AUTO_POSTER_DIR . '/includes/shorteners/bitly.php' );
				//get bitly user name & api key
				$wpw_auto_poster_bitly_username = $wpw_auto_poster_options[$socialtype.'_bitly_username'];
				$wpw_auto_poster_bitly_api_key = $wpw_auto_poster_options[$socialtype.'_bitly_api_key'];
				
				$bitlyurl = new wpw_auto_poster_tw_bitly( $wpw_auto_poster_bitly_username, $wpw_auto_poster_bitly_api_key );
				$link = $bitlyurl->shorten( $link );
				break;
		}
		return $link;
	}
	/**
	 * Replace Shortcode In Twitter Template
	 * 
	 * Handles to return template with replace its shortcodes
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	public function wpw_auto_poster_tweet_status( $post, $template, $title = '' ) {
			
		global $wpw_auto_poster_options;
		
		$prefix = WPW_AUTO_POSTER_META_PREFIX;	
		
		$ispublished = get_post_meta( $post->ID, $prefix . 'tw_status', true );
		$unique = 'false';
		
		if( isset( $ispublished ) && $ispublished == '1' ) { //if post is published on facebook once then change url to prevent duplication
			//unique link for posting
			$unique = 'true';
		}
		
		$postlink = get_permalink( $post->ID );	
		$postlink = $this->wpw_auto_poster_get_short_post_link( $postlink, $unique, $post->ID, 'false', 'tw' );
		
		$posttitle = $post->post_title;
		$userdata = get_userdata($post->post_author);
		$nicename = $userdata->user_nicename;
		$first_name = get_user_meta( $post->post_author, 'first_name', true);
		$last_name = get_user_meta( $post->post_author, 'last_name', true);
		$fullauthor = $first_name . ' ' . $last_name;
		$posttype = $post->post_type;
		
		//if title is passed from function parameter then use that title
		$posttitle = !empty($title) ? $title : $posttitle;
		
		$replacetags = array( '[link]','[title]', '[full_author]', '[nickname_author]', '[post_type]');
		$replaceval = array( $postlink, $posttitle, $fullauthor, $nicename, $posttype );
		$status = str_replace( $replacetags, $replaceval, $template );
		
		$replacetags = array( '{link}','{title}', '{full_author}', '{nickname_author}', '{post_type}');
		$replaceval = array( $postlink, $posttitle, $fullauthor, $nicename, $posttype );
		$status = str_replace( $replacetags, $replaceval, $status );
		
		return $this->wpw_auto_poster_html_decode( $status );
	}
	/**
	 * Return Template Text from Value
	 * 
	 * Handles to return Template
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	public function wpw_auto_poster_get_tweet_template($tweet_template) {
	
		global $wpw_auto_poster_options;
		$retval = '';
		
		switch ( $tweet_template ) {
			
			case 'title_fullauthor_link' : 
					$retval = '[title] by [full_author] - [link]';
					break;
			
			case 'title_nickname_link'	:
					$retval = '[title] by @[nickname_author] - [link]';
					break;
					
			case 'post_type_title_link'	:
					$retval = 'New [post_type]: [title] - [link]';
					break;
			
			case 'post_type_title_fullauthor_link'	:
					$retval = 'New [post_type]: [title] by [full_author] - [link]';
					break;
					
			case 'post_type_title_nickname_link' :
					$retval = 'New [post_type]: [title] by [nickname_author] - [link]';
					break;
			case 'custom' 	:
					$retval = $wpw_auto_poster_options['tw_custom_tweet_template'];
					break;
			case  'title_link' 	:
			default :
					$retval = '[title] - [link]';
					break;
		}
		
		return $retval;
	
	}
	
	/**
	 * Get Self URL
	 * 
	 * Handles to return current URL
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	public function wpw_auto_poster_self_url() 
	{ 
	    $s = empty($_SERVER["HTTPS"]) ? '' : ($_SERVER["HTTPS"] == "on") ? "s" : ""; 
	    $str1 = strtolower($_SERVER["SERVER_PROTOCOL"]);
	    $str2 = "/";
	    $protocol = substr($str1, 0, strpos($str1, $str2)).$s;
	    $port = ($_SERVER["SERVER_PORT"] == "80") ? "" : (":".$_SERVER["SERVER_PORT"]); 
	    return $protocol."://".$_SERVER['SERVER_NAME'].$port.$_SERVER['REQUEST_URI']; 
	} 
	/**
	 * Short the Content As Per Character Limit
	 * 
	 * Handles to return short content as per character 
	 * limit
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 **/
	public function wpw_auto_poster_excerpt( $content, $charlength = 140 ) {
		
		$excerpt = '';
		$charlength++;
		
		//check content length is greater then character length
		if( strlen( $content ) > $charlength ) {
			
			$subex 		=	substr( $content, 0, $charlength - 5 );
			$exwords 	=	explode( ' ', $subex );
			$excut 		=-	( strlen( $exwords[ count( $exwords ) - 1 ] ) );
			
			if ( $excut < 0 ) {
				$excerpt = substr( $subex, 0, $excut );
			} else {
				$excerpt = $subex;
			}
		} else {
			$excerpt = $content;
		}
		
		//return short content
		return $excerpt;
	}
	
	/**
	 * Insert Social Posting Log
	 * 
	 * Handles to insert social posting log
	 * 
	 * @package Social Auto Poster
	 * @since 1.4.0
	 **/
	public function wpw_auto_poster_insert_posting_log( $post_id, $social_type, $posting_logs_data = array(), $posting_logs_user_details = array() ) {
		
		global $wpw_auto_poster_options, $current_user;
		
		// Check Enable Posting Logs is enable from general settings
		// Check post id, social type and posting data are not empty
		if( isset( $wpw_auto_poster_options['enable_posting_logs'] ) && $wpw_auto_poster_options['enable_posting_logs'] == '1'
			&& !empty( $post_id ) && !empty( $social_type ) && !empty( $posting_logs_data ) ) {
			
			$prefix = WPW_AUTO_POSTER_META_PREFIX;
			
			$userid = '0';
			if( is_user_logged_in() ) { // Check user is logged in
				$userid = $current_user->ID;
			}
			
			//create array arguments for saving the social posting data to database
			$posting_args = array(
										  'post_title'		=>	'',
										  'post_content'	=>	'',
										  'post_status'		=>	'publish',
										  'post_type'		=>	WPW_AUTO_POSTER_LOGS_POST_TYPE,
										  'post_parent'		=>	$post_id,
										  'post_author'		=>	$userid
									);
			
			// insert the social posting data to database
			$postingid = wp_insert_post( $posting_args );
			
			//if social posting basic data is successfully stored then update some more data to database
			if( !empty( $postingid ) ) { //check record is inserted in database
				
				// update social type
				update_post_meta( $postingid, $prefix . 'social_type', $social_type );
				
				// update social posting data
				update_post_meta( $postingid, $prefix . 'posting_logs', $posting_logs_data );
				
				// update social posting user details
				update_post_meta( $postingid, $prefix . 'user_details', $posting_logs_user_details );
			}
		}
	}
	
	/**
	 * Convert Object To Array
	 *
	 * Converting Object Type Data To Array Type
	 * 
	 * @package Social Auto Poster
	 * @since 1.4.0
	 */
	public function wpw_auto_poster_object_to_array( $result )
	{
	    $array = array();
	    foreach ($result as $key=>$value)
	    {	
	        if (is_object($value))
	        {
	            $array[$key]=$this->wpw_auto_poster_object_to_array($value);
	        } else {
	        	$array[$key]=$value;
	        }
	    }
	    return $array;
	}
	
	/**
	 * Get Posting Logs data
	 * 
	 * Handles to get posting logs data
	 * 
	 * @package Social Auto Poster
	 * @since 1.4.0
	 */
	public function wpw_auto_poster_get_posting_logs_data( $args = array() ) {
		
		$prefix = WPW_AUTO_POSTER_META_PREFIX;
		
		$postinglogsargs = array( 
										'post_type' 		=> WPW_AUTO_POSTER_LOGS_POST_TYPE,
										'post_status' 		=> 'publish'
									);

		$postinglogsargs = wp_parse_args( $args, $postinglogsargs );

		//show how many per page records
		if( isset( $args['posts_per_page'] ) && !empty( $args['posts_per_page'] ) ) {
			$postinglogsargs['posts_per_page'] = $args['posts_per_page'];
		} else {
			$postinglogsargs['posts_per_page'] = '-1';
		}
		
		//show per page records
		if( isset( $args['paged']) && !empty( $args['paged'] ) ) {
			$postinglogsargs['paged'] = $args['paged'];
		}
		
		//if search using user
		if( isset( $args['author'] ) ) {
			$postinglogsargs['author'] = $args['author'];
		}
		
		//if search using post parent
		if( isset( $args['post_parent'] ) ) {
			$postinglogsargs['post_parent'] = $args['post_parent'];
		}
		
		//if serch using meta query
		if( isset( $args['meta_query'] ) ) {
			$postinglogsargs['meta_query'] = $args['meta_query'];
		}
		
		//if returns only id
		if( isset( $args['fields'] ) && !empty( $args['fields'] ) ) {
			$postinglogsargs['fields'] = $args['fields'];
		}
		
		//if search is called then retrive searching data
		if( isset( $args['search'] ) ) {
			$postinglogsargs['s'] = $args['search'];
		}
		
		//if month is called then retrive monthly data
		if( isset( $args['m'] ) ) {
			$postinglogsargs['m'] = $args['m'];
		}
		
		//get order by records
		$postinglogsargs['order'] = 'DESC';
		$postinglogsargs['orderby'] = 'date';

		//show order by field records
		if(isset($args['order']) && !empty($args['order'])) {
			$postinglogsargs['order'] = $args['order'];
		}

		//show order by order records
		if(isset($args['orderby']) && !empty($args['orderby'])) {
			$postinglogsargs['orderby'] = $args['orderby'];
		}

		//fire query in to table for retriving data
		$result = new WP_Query( $postinglogsargs );
	
		if( isset( $args['count'] ) && $args['count'] == '1') {
			$postinglogslist = $result->post_count;	
		}  else {
			//retrived data is in object format so assign that data to array for listing
			$postinglogslist = $this->wpw_auto_poster_object_to_array($result->posts);

			// if get list for auto poster list list then return data with data and total array
			if( isset($args['wpw_auto_poster_list']) && $args['wpw_auto_poster_list'] ) {

				$data_res	= array();

				$data_res['data'] 	= $postinglogslist;

				//To get total count of post using "found_posts" and for users "total_users" parameter
				$data_res['total']	= isset($result->found_posts) ? $result->found_posts : '';

				return $data_res;
			}
		}
		
		return $postinglogslist;
	}
	
	/**
	 * Bulk Deletion
	 *
	 * Does handle deleting posts from the
	 * database table.
	 *
	 * @package Social Auto Poster
	 * @since 1.4.0
	 */
	public function wpw_auto_poster_bulk_delete( $args = array() ) { 
   
   		global $wpdb;
		
		if( isset( $args['log_id'] ) && !empty( $args['log_id'] ) ) {
		
			wp_delete_post( $args['log_id'] );
		}
	}
	
	/**
	 * Get Date Format
	 * 
	 * Handles to return formatted date which format is set in backend
	 * 
	 * @package Social Auto Poster
	 * @since 1.4.0
	 */
	public function wpw_auto_poster_get_date_format( $date, $time = false ) {
		
		$format = $time ? get_option( 'date_format' ).' '.get_option('time_format') : get_option('date_format');
		$date = date_i18n( $format, strtotime($date));
		return apply_filters( 'wpw_auto_poster_date_format', $date );
	}
	
	/**
	 * All Social Types
	 * 
	 * Handles to return all social types
	 * 
	 * @package Social Auto Poster
	 * @since 1.4.0
	 */
	public function wpw_auto_poster_get_social_type_name( $social_type = '' ) {
		
		$social_types = array(
									'fb' => __( 'Facebook', 'wpwautoposter' ),
									'tw' => __( 'Twitter', 'wpwautoposter' ),
									'li' => __( 'LinkedIn', 'wpwautoposter' ),
									'tb' => __( 'Tumblr', 'wpwautoposter' ),
									'dc' => __( 'Delicious', 'wpwautoposter' ),
									'ba' => __( 'BufferApp', 'wpwautoposter' ),
								);
		
		if( empty( $social_type ) ) {
			
			return $social_types;
		}
		return !empty( $social_type ) && isset( $social_types[$social_type] ) ? $social_types[$social_type] : '';
	}
	
	/**
	 * Get Facebook Posting Method
	 * 
	 * Handles to get facebook posting method
	 * 
	 * @package Social Auto Poster
	 * @since 1.4.0
	 */
	public function wpw_auto_poster_get_fb_posting_method( $method_key = '' ) {
		
		$fb_posting_method = array(
									'feed' 			=> __( 'As a Wall Post', 'wpwautoposter' ),
									'feed_status' 	=> __( 'As a Status Update', 'wpwautoposter' ),
									//'notes' 		=> __( 'As a Note', 'wpwautoposter' ),
									//'events' 		=> __( 'As an Event', 'wpwautoposter' )
								);
		
		if( empty( $method_key ) ) { // Check method is empty 
			
			return $fb_posting_method;
		}
		return !empty( $method_key ) && isset( $fb_posting_method[$method_key] ) ? $fb_posting_method[$method_key] : '';
	}
	
	/**
	 * Get One Diemention Array
	 * 
	 * Handles to get one diemention array by two diemention array
	 * 
	 * @package Social Auto Poster
	 * @since 1.4.0
	 */
	public function wpw_auto_poster_get_one_dim_array( $multi_dim_array ) {
		
		$one_dim_array = array();
		if( !empty( $multi_dim_array ) ) { // Check dim array are not empty
			
			foreach( $multi_dim_array as $multi_dim_keys ) {
	    
				if( !empty( $multi_dim_keys ) ) { // Check dim keys are not empty
					
					foreach( $multi_dim_keys as $multi_dim_values ) {
						
			     		$one_dim_array[] = $multi_dim_values;
					}
				}
			}
		}
		return $one_dim_array;
	}
	
	/**
	 * Get All Schedules
	 * 
	 * Handle to get all schedules
	 *
	 * @package Social Auto Poster
	 * @since 1.5.0
	 */
	public function wpw_auto_poster_get_all_schedules() {
		
		$all_schedules = array( 
									'' 			=> __( 'Instantly', 'wpwautoposter' ),
									'hourly' 	=> __( 'Hourly', 'wpwautoposter' ),
									'twicedaily'=> __( 'Twice Daily', 'wpwautoposter' ),
									'daily' 	=> __( 'Daily', 'wpwautoposter' ),
									'weekly' 	=> __( 'Weekly', 'wpwautoposter' )
								);
		return $all_schedules;
	}
	
	/**
	 * Get All Schedule Time
	 * 
	 * Handle to get all schedule time
	 *
	 * @package Social Auto Poster
	 * @since 1.5.0
	 */
	public function wpw_auto_poster_get_all_schedule_time() {
		
		$all_schedules = array(
									'0'	=> __( '12 AM', 'wpwautoposter' ),
									'1'	=> __( '1 AM', 'wpwautoposter' ),
									'2'	=> __( '2 AM', 'wpwautoposter' ),
									'3'	=> __( '3 AM', 'wpwautoposter' ),
									'4'	=> __( '4 AM', 'wpwautoposter' ),
									'5'	=> __( '5 AM', 'wpwautoposter' ),
									'6'	=> __( '6 AM', 'wpwautoposter' ),
									'7'	=> __( '7 AM', 'wpwautoposter' ),
									'8'	=> __( '8 AM', 'wpwautoposter' ),
									'9'	=> __( '9 AM', 'wpwautoposter' ),
									'10'=> __( '10 AM', 'wpwautoposter' ),
									'11'=> __( '11 AM', 'wpwautoposter' ),
									'12'=> __( '12 PM', 'wpwautoposter' ),
									'13'=> __( '1 PM', 'wpwautoposter' ),
									'14'=> __( '2 PM', 'wpwautoposter' ),
									'15'=> __( '3 PM', 'wpwautoposter' ),
									'16'=> __( '4 PM', 'wpwautoposter' ),
									'17'=> __( '5 PM', 'wpwautoposter' ),
									'18'=> __( '6 PM', 'wpwautoposter' ),
									'19'=> __( '7 PM', 'wpwautoposter' ),
									'20'=> __( '8 PM', 'wpwautoposter' ),
									'21'=> __( '9 PM', 'wpwautoposter' ),
									'22'=> __( '10 PM', 'wpwautoposter' ),
									'23'=> __( '11 PM', 'wpwautoposter' ),
								);
		return $all_schedules;
	}
	
	/**
	 * Get All Schedule Minutes
	 * 
	 * Handle to get all schedule minutes
	 *
	 * @package Social Auto Poster
	 * @since 1.5.2
	 */
	public function wpw_auto_poster_get_all_schedule_minutes() {
		
		$all_schedules = array( 
									'0'		=> __( '00 Min', 'wpwautoposter' ),
									'5'		=> __( '05 Min', 'wpwautoposter' ),
									'10'	=> __( '10 Min', 'wpwautoposter' ),
									'15'	=> __( '15 Min', 'wpwautoposter' ),
									'20'	=> __( '20 Min', 'wpwautoposter' ),
									'25'	=> __( '25 Min', 'wpwautoposter' ),
									'30'	=> __( '30 Min', 'wpwautoposter' ),
									'35'	=> __( '35 Min', 'wpwautoposter' ),
									'40'	=> __( '40 Min', 'wpwautoposter' ),
									'45'	=> __( '45 Min', 'wpwautoposter' ),
									'50'	=> __( '50 Min', 'wpwautoposter' ),
									'55'	=> __( '55 Min', 'wpwautoposter' ),
								);
		return $all_schedules;
	}
	
	/**
	 * Get All Post Data
	 * 
	 * Handle to get all post data which
	 * have send mails to followers
	 *
	 * @package Social Auto Poster
	 * @since 1.5.0
	 */
	public function wpw_auto_poster_get_schedule_post_data() {

		$prefix = WPW_AUTO_POSTER_META_PREFIX;
		
		$postargs = array( 
								'post_type' 	=> 'any',
								'posts_per_page'=> '-1',
								'meta_query'	=> array(
															array(
																		'key'		=> $prefix . 'schedule_wallpost',
																		'value'		=> '',
																		'compare'	=> '!='
																	)
														)
							);
		
		//fire query in to table for retriving data
		$result = new WP_Query( $postargs );
		
		$postslist = $result->posts;
		
		// Check if post type counter from filter
		if( isset( $args['count'] ) && !empty( $args['count'] ) ) {
			return count( $postslist );
		}
		
		return $postslist;
	}
}